<?php

require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use App\Models\Warehouse\ProjectInventory;
use App\Models\DirectDelivery;

try {
    echo "=== Direct Delivery Project Inventory Sync Script ===" . PHP_EOL;
    echo "Syncing delivered direct deliveries with project inventory..." . PHP_EOL . PHP_EOL;

    // Get all delivered direct deliveries that should update project inventory
    $deliveredDirectDeliveries = DB::table('direct_deliveries')
        ->where('status', 'delivered')
        ->whereNotNull('project_id')
        ->get();

    echo "Found " . $deliveredDirectDeliveries->count() . " delivered direct deliveries to process..." . PHP_EOL . PHP_EOL;

    if ($deliveredDirectDeliveries->isEmpty()) {
        echo "No delivered direct deliveries found to sync." . PHP_EOL;
        exit;
    }

    $processedCount = 0;
    $createdRecords = 0;
    $updatedRecords = 0;
    $skippedCount = 0;
    $currentDelivery = null;

    foreach ($deliveredDirectDeliveries as $delivery) {
        if ($currentDelivery !== $delivery->id) {
            echo PHP_EOL . "--- Processing Direct Delivery ID: {$delivery->id} (Project: {$delivery->project_id}) ---" . PHP_EOL;
            $currentDelivery = $delivery->id;
        }

        // Get items for this delivery
        $deliveryItems = DB::table('direct_delivery_items')
            ->where('direct_delivery_id', $delivery->id)
            ->get();

        foreach ($deliveryItems as $item) {
            // For delivered items, use delivered_quantity if received_quantity is 0
            $quantityToProcess = $item->received_quantity > 0 ? $item->received_quantity : $item->delivered_quantity;

            if ($quantityToProcess <= 0) {
                echo "  - Skipped Item {$item->item_id}: No quantity to process" . PHP_EOL;
                $skippedCount++;
                continue;
            }

            echo "Processing Item ID: {$item->item_id}" . PHP_EOL;

            // Check if project inventory record exists
            $projectInventory = ProjectInventory::where('project_id', $delivery->project_id)
                ->where('item_id', $item->item_id)
                ->first();

            if ($projectInventory) {
                // Update existing record
                $oldQuantity = $projectInventory->quantity_available;
                $projectInventory->updateQuantity($quantityToProcess, 'add');
                $newQuantity = $projectInventory->quantity_available;

                echo "  - Updated: Qty {$oldQuantity} + {$quantityToProcess} = {$newQuantity}" . PHP_EOL;
                $updatedRecords++;
            } else {
                // Create new record
                $projectInventory = ProjectInventory::create([
                    'project_id' => $delivery->project_id,
                    'item_id' => $item->item_id,
                    'quantity_available' => $quantityToProcess,
                    'allocated_quantity' => 0,
                    'unit_price' => $item->unit_price ?: 0,
                    'total_value' => $quantityToProcess * ($item->unit_price ?: 0)
                ]);

                echo "  - Created: Qty {$quantityToProcess} @ Price " . ($item->unit_price ?: 0) . PHP_EOL;
                $createdRecords++;
            }

            // Update the received_quantity if it's 0 (for future consistency)
            if ($item->received_quantity == 0 && $item->delivered_quantity > 0) {
                DB::table('direct_delivery_items')
                    ->where('id', $item->id)
                    ->update([
                        'received_quantity' => $item->delivered_quantity,
                        'quality_status' => 'passed'
                    ]);
                echo "  - Updated delivery item received_quantity for consistency" . PHP_EOL;
            }

            $processedCount++;
        }
    }

    echo PHP_EOL . "=== Sync Complete ===" . PHP_EOL;
    echo "Total items processed: {$processedCount}" . PHP_EOL;
    echo "New project inventory records created: {$createdRecords}" . PHP_EOL;
    echo "Existing records updated: {$updatedRecords}" . PHP_EOL;
    echo "Items skipped (no quantity): {$skippedCount}" . PHP_EOL;

    // Display updated project inventory summary
    echo PHP_EOL . "=== Updated Project Inventory Summary ===" . PHP_EOL;
    $projectInventorySummary = ProjectInventory::select(
        'project_id',
        DB::raw('COUNT(*) as item_count'),
        DB::raw('SUM(quantity_available) as total_quantity'),
        DB::raw('SUM(total_value) as total_value')
    )
    ->groupBy('project_id')
    ->orderBy('project_id')
    ->get();

    foreach ($projectInventorySummary as $summary) {
        echo "Project {$summary->project_id}: {$summary->item_count} items, Total Qty: {$summary->total_quantity}, Total Value: " . number_format($summary->total_value, 2) . " AED" . PHP_EOL;
    }

} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . PHP_EOL;
    echo "Trace: " . $e->getTraceAsString() . PHP_EOL;
}