@extends('layouts.admin-simple')

@section('title', 'New Material Transfer Request')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.transfer-requests.index') }}">Transfer Requests</a></li>
    <li class="breadcrumb-item active">New Request</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">New Material Transfer Request</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Request material transfer between projects</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <a href="{{ route('warehouse.transfer-requests.index') }}" class="btn btn-sm btn-outline-secondary mr-2">
                <i class="material-icons mr-1" style="font-size: 16px;">arrow_back</i>
                Back to List
            </a>
        </div>
    </div>
@endsection

@section('content')
    <form id="transferForm" action="{{ route('warehouse.transfer-requests.store') }}" method="POST">
        @csrf

        <!-- Validation Errors -->
        @if ($errors->any())
            <div class="alert alert-danger">
                <h6><i class="material-icons align-middle mr-1">error</i> Please fix the following errors:</h6>
                <ul class="mb-0">
                    @foreach ($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
        @endif

        <!-- Basic Information Card -->
        <div class="card mb-3">
            <div class="card-header">
                <h6 class="card-title mb-0 d-flex align-items-center">
                    <i class="material-icons mr-2">info</i>
                    Project Information
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="transferrer_project_id">From Project <span class="text-danger">*</span></label>
                            <select name="transferrer_project_id" id="transferrer_project_id" class="form-control @error('transferrer_project_id') is-invalid @enderror" required>
                                <option value="">Select Project</option>
                                @foreach($projects as $project)
                                    <option value="{{ $project->id }}" {{ old('transferrer_project_id') == $project->id ? 'selected' : '' }}>
                                        {{ $project->project_name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('transferrer_project_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="receiver_project_id">To Project <span class="text-danger">*</span></label>
                            <select name="receiver_project_id" id="receiver_project_id" class="form-control @error('receiver_project_id') is-invalid @enderror" required>
                                <option value="">Select Project</option>
                                @foreach($projects as $project)
                                    <option value="{{ $project->id }}" {{ old('receiver_project_id') == $project->id ? 'selected' : '' }}>
                                        {{ $project->project_name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('receiver_project_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="transfer_reason">Transfer Reason <span class="text-danger">*</span></label>
                            <input type="text" name="transfer_reason" id="transfer_reason"
                                   class="form-control @error('transfer_reason') is-invalid @enderror"
                                   value="{{ old('transfer_reason') }}"
                                   placeholder="e.g., Material shortage on receiver project" required>
                            @error('transfer_reason')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="priority">Priority <span class="text-danger">*</span></label>
                            <select name="priority" id="priority" class="form-control @error('priority') is-invalid @enderror" required>
                                <option value="medium" {{ old('priority', 'medium') == 'medium' ? 'selected' : '' }}>Medium</option>
                                <option value="low" {{ old('priority') == 'low' ? 'selected' : '' }}>Low</option>
                                <option value="high" {{ old('priority') == 'high' ? 'selected' : '' }}>High</option>
                                <option value="urgent" {{ old('priority') == 'urgent' ? 'selected' : '' }}>Urgent</option>
                            </select>
                            @error('priority')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="required_date">Required Date</label>
                            <input type="date" name="required_date" id="required_date"
                                   class="form-control @error('required_date') is-invalid @enderror"
                                   value="{{ old('required_date') }}"
                                   min="{{ date('Y-m-d', strtotime('tomorrow')) }}">
                            @error('required_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="requested_by">Requested By <span class="text-danger">*</span></label>
                            <select name="requested_by" id="requested_by" class="form-control @error('requested_by') is-invalid @enderror" required>
                                @foreach($users as $user)
                                    <option value="{{ $user->id }}" {{ old('requested_by', auth()->id()) == $user->id ? 'selected' : '' }}>
                                        {{ $user->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('requested_by')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="request_date">Request Date <span class="text-danger">*</span></label>
                            <input type="date" name="request_date" id="request_date"
                                   class="form-control @error('request_date') is-invalid @enderror"
                                   value="{{ old('request_date', date('Y-m-d')) }}" required>
                            @error('request_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>
        </div>


        <!-- Transfer Items Card -->
        <div class="card mb-3">
            <div class="card-header">
                <h6 class="card-title mb-0 d-flex align-items-center">
                    <i class="material-icons mr-2">swap_horiz</i>
                    Transfer Items
                </h6>
            </div>
            <div class="card-body p-0">
                <div id="no-materials-message" class="text-center py-5" style="display: none;">
                    <i class="material-icons text-muted mb-2" style="font-size: 48px;">inventory</i>
                    <h5 class="text-muted">No Items Available</h5>
                    <p class="text-muted">No items available in the selected project inventory.</p>
                </div>

                <div class="table-responsive" id="materials-table-container">
                    <table class="table table-sm mb-0" id="materials-table">
                        <thead class="bg-light">
                            <tr>
                                <th style="width: 5%">
                                    <input type="checkbox" id="selectAll" onchange="toggleAllItems()">
                                </th>
                                <th style="width: 30%">Item Description</th>
                                <th style="width: 10%">Item Code</th>
                                <th style="width: 8%">Unit</th>
                                <th style="width: 12%">Available Qty</th>
                                <th style="width: 15%">Transfer Qty</th>
                                <th style="width: 20%">Notes</th>
                            </tr>
                        </thead>
                        <tbody id="materials-tbody">
                            <!-- Items will be loaded from project inventory -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Additional Information Card -->
        <div class="card mb-3">
            <div class="card-header">
                <h6 class="card-title mb-0 d-flex align-items-center">
                    <i class="material-icons mr-2">note</i>
                    Additional Information
                </h6>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label for="transfer_notes">Transfer Notes</label>
                    <textarea name="transfer_notes" id="transfer_notes" rows="3"
                              class="form-control @error('transfer_notes') is-invalid @enderror"
                              placeholder="Any additional information or special instructions">{{ old('transfer_notes') }}</textarea>
                    @error('transfer_notes')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>
            </div>
        </div>

        <!-- Form Actions -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="text-muted">
                                <small>Complete all required fields above to submit your transfer request</small>
                            </div>
                            <div>
                                <button type="submit" name="action" value="save_draft" class="btn btn-secondary mr-2">
                                    <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                                    Save as Draft
                                </button>
                                <button type="submit" name="action" value="submit" class="btn btn-primary">
                                    <i class="material-icons mr-1" style="font-size: 16px;">send</i>
                                    Submit for Approval
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

    </form>
@endsection

@push('styles')
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css">
<style>
    .select2-container .select2-selection--single {
        height: 38px;
        border: 1px solid #ced4da;
    }

    .select2-container--default .select2-selection--single .select2-selection__rendered {
        line-height: 36px;
    }

    .select2-container--default .select2-selection--single .select2-selection__arrow {
        height: 36px;
    }

    .form-group label {
        font-weight: 600;
        color: #495057;
        font-size: 14px;
    }

    .table th {
        border-top: none;
        font-weight: 600;
        color: #495057;
        background-color: #f8f9fa;
    }
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
$(document).ready(function() {
    // Initialize Select2
    $('#transferrer_project_id, #receiver_project_id, #priority, #requested_by').select2({
        placeholder: 'Please select...',
        allowClear: true
    });

    // Handle transferrer project selection - load project inventory directly
    $('#transferrer_project_id').change(function() {
        const projectId = $(this).val();
        if (projectId) {
            loadProjectInventory(projectId);
        } else {
            clearMaterialsData();
        }
    });
});

function loadProjectInventory(projectId) {
    console.log('Loading inventory for project ID:', projectId);

    const ajaxUrl = "{{ route('warehouse.transfer-requests.project-inventory', ':id') }}".replace(':id', projectId);

    $.ajax({
        url: ajaxUrl,
        method: 'GET',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
            'Accept': 'application/json'
        },
        success: function(inventory) {
            console.log('Project inventory loaded:', inventory);
            populateItemsTable(inventory);
        },
        error: function(xhr, status, error) {
            console.error('Error loading project inventory:', xhr.responseText);
            clearMaterialsData();
            alert('Error loading project inventory. Please try again.');
        }
    });
}

function populateItemsTable(inventory) {
    const tbody = document.getElementById('materials-tbody');
    tbody.innerHTML = '';

    if (!inventory || inventory.length === 0) {
        document.getElementById('no-materials-message').style.display = 'block';
        document.getElementById('materials-table-container').style.display = 'none';
        return;
    }

    inventory.forEach(function(item, index) {
        const row = createItemRow(item, index);
        tbody.appendChild(row);
    });

    document.getElementById('no-materials-message').style.display = 'none';
    document.getElementById('materials-table-container').style.display = 'block';
}

function createItemRow(item, index) {
    const row = document.createElement('tr');
    row.className = 'material-row';
    row.setAttribute('data-row', index);

    const availableQty = item.available_for_transfer || item.quantity_available || 0;

    row.innerHTML = `
        <td>
            <input type="checkbox" class="item-checkbox" data-index="${index}" onchange="toggleItemRow(${index})">
        </td>
        <td>
            <strong>${item.item_name || 'N/A'}</strong>
        </td>
        <td><code>${item.item?.item_code || 'N/A'}</code></td>
        <td><span class="badge badge-secondary">${item.item?.unit_of_measure || 'PCS'}</span></td>
        <td class="text-center">
            <strong class="text-success">${parseFloat(availableQty).toFixed(2)}</strong>
        </td>
        <td>
            <input type="number" name="items[${index}][quantity_requested]"
                   class="form-control form-control-sm quantity-transfer"
                   value="0" step="0.01" min="0" max="${availableQty}"
                   data-max="${availableQty}"
                   disabled
                   onchange="updateTransferRowVisual(${index})">
            <input type="hidden" name="items[${index}][item_id]" value="${item.item_id}">
        </td>
        <td>
            <input type="text" name="items[${index}][transfer_notes]"
                   class="form-control form-control-sm"
                   placeholder="Transfer notes"
                   disabled>
        </td>
    `;

    return row;
}

function toggleItemRow(index) {
    const row = document.querySelector(`tr[data-row="${index}"]`);
    const checkbox = row.querySelector('.item-checkbox');
    const quantityInput = row.querySelector('.quantity-transfer');
    const notesInput = row.querySelector('input[name*="[transfer_notes]"]');

    if (checkbox.checked) {
        quantityInput.disabled = false;
        notesInput.disabled = false;
        quantityInput.value = quantityInput.getAttribute('data-max');
        row.classList.add('table-success');
    } else {
        quantityInput.disabled = true;
        notesInput.disabled = true;
        quantityInput.value = 0;
        row.classList.remove('table-success');
    }
}

function toggleAllItems() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.item-checkbox');

    checkboxes.forEach((checkbox, index) => {
        checkbox.checked = selectAll.checked;
        toggleItemRow(index);
    });
}

function updateTransferRowVisual(index) {
    const row = document.querySelector(`tr[data-row="${index}"]`);
    const quantityInput = row.querySelector('input[name*="[quantity_requested]"]');
    const quantity = parseFloat(quantityInput.value) || 0;

    // Add visual feedback for rows with transfer quantities
    if (quantity > 0) {
        row.classList.add('table-success');
        row.classList.remove('table-light');
    } else {
        row.classList.remove('table-success');
        row.classList.add('table-light');
    }
}

function clearMaterialsData() {
    document.getElementById('materials-tbody').innerHTML = '';
    document.getElementById('no-materials-message').style.display = 'block';
    document.getElementById('materials-table-container').style.display = 'none';
}

// Form validation before submit
document.getElementById('transferForm').addEventListener('submit', function(e) {
    const checkedItems = document.querySelectorAll('.item-checkbox:checked');

    if (checkedItems.length === 0) {
        e.preventDefault();
        alert('Please select at least one item to transfer.');
        return false;
    }

    // Remove unchecked items from submission
    document.querySelectorAll('.material-row').forEach(function(row) {
        const checkbox = row.querySelector('.item-checkbox');
        if (!checkbox.checked) {
            row.remove();
        }
    });
});
</script>
@endpush