@extends('layouts.admin-simple')

@section('title', 'Site Return Operations')

{{-- Force cache refresh --}}
<!-- Last updated: {{ now() }} -->

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Site Return</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Site Return Operations</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage returns from construction sites</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="exportReturns()" class="btn btn-sm btn-info mr-2" title="Export returns to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>

            <a href="{{ route('warehouse.site-return.create') }}" class="btn btn-sm btn-primary" title="Add new site return">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                New Return
            </a>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Projects with Returns"
                :value="$statistics['total_projects_with_returns'] ?? 0"
                icon="business"
                color="warning"
                subtitle="Active projects"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Returns"
                :value="$statistics['total_returns'] ?? 0"
                icon="assignment_return"
                color="primary"
                subtitle="All site returns"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Material Requests"
                :value="$statistics['total_material_requests'] ?? 0"
                icon="assignment"
                color="success"
                subtitle="With returns"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Items Returned"
                :value="number_format($statistics['total_items_returned'] ?? 0)"
                icon="inventory_2"
                color="info"
                subtitle="Total items"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-3">
                    @php
                        $projectFilterOptions = [['value' => '', 'text' => 'All Projects']];
                        foreach($allProjects ?? [] as $project) {
                            $projectFilterOptions[] = [
                                'value' => $project->id,
                                'text' => ($project->project_number ? $project->project_number . ' - ' : '') . $project->project_name
                            ];
                        }
                    @endphp
                    <x-warehouse.select2-dropdown
                        name="filter-project"
                        id="filter-project"
                        label="Project"
                        placeholder="All Projects"
                        :options="$projectFilterOptions"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Search Project</label>
                    <input type="text" id="filter-search" class="form-control" placeholder="Search project name...">
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    @php
                        $divisionFilterOptions = [['value' => '', 'text' => 'All Divisions']];
                        $uniqueDivisions = $allProjects->unique('projectDivision.division_name')->filter(function($project) {
                            return $project->projectDivision;
                        });
                        foreach($uniqueDivisions as $project) {
                            $divisionFilterOptions[] = [
                                'value' => $project->projectDivision->division_name,
                                'text' => $project->projectDivision->division_name
                            ];
                        }
                    @endphp
                    <x-warehouse.select2-dropdown
                        name="filter-division"
                        id="filter-division"
                        label="Division"
                        placeholder="All Divisions"
                        :options="$divisionFilterOptions"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter-sort"
                        id="filter-sort"
                        label="Sort By"
                        placeholder="Select Sort Order"
                        :options="[
                            ['value' => 'latest_return', 'text' => 'Latest Return Date'],
                            ['value' => 'project_name', 'text' => 'Project Name'],
                            ['value' => 'most_returns', 'text' => 'Most Returns'],
                            ['value' => 'total_items', 'text' => 'Total Items']
                        ]"
                        selected="latest_return"
                        allowClear="false"
                    />
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Site Returns Table -->
    <div class="card">
        <div class="card-header warehouse-report-header">
            <h5 class="mb-0 warehouse-report-title">
                <i class="material-icons align-middle mr-2">assignment_return</i>
                Projects with Site Returns
            </h5>
        </div>
        <div class="card-body p-0">
            @if(isset($projects) && $projects->count() > 0)
                <div class="warehouse-table-container">
                    <!-- Loading Overlay -->
                    <div class="warehouse-loading-overlay" id="tableLoadingOverlay" style="display: none;">
                        <div class="warehouse-loading-content">
                            <div class="text-center">
                                <div class="spinner-border text-primary" role="status">
                                    <span class="sr-only">Loading...</span>
                                </div>
                                <div class="mt-2">Loading returns...</div>
                            </div>
                        </div>
                    </div>

                    <!-- Table -->
                    <div class="warehouse-table-wrapper">
                        <table class="table table-hover mb-0 warehouse-table">
                        <thead class="thead-light">
                            <tr>
                                <th style="min-width: 200px;">Project</th>
                                <th style="min-width: 120px;">Division</th>
                                <th style="min-width: 120px;">Total Returns</th>
                                <th style="min-width: 140px;">Material Requests</th>
                                <th style="min-width: 140px;">Latest Return Date</th>
                                <th style="min-width: 100px;">Status</th>
                                <th style="min-width: 120px;">Items Returned</th>
                                <th style="min-width: 140px;">Total Value (AED)</th>
                                <th style="min-width: 200px;">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($projects as $project)
                                <tr>
                                    <td style="white-space: nowrap;">
                                        <div class="d-flex align-items-center">
                                            <div class="mr-3">
                                                <div class="bg-info text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                                    <i class="material-icons" style="font-size: 18px;">business</i>
                                                </div>
                                            </div>
                                            <div>
                                                <div class="font-weight-medium">{{ $project->project_name }}</div>
                                                <small class="text-muted">{{ $project->project_number ?: 'No project number' }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge badge-secondary">{{ $project->project_division ?: 'N/A' }}</span>
                                    </td>
                                    <td>
                                        <span class="badge badge-primary">{{ $project->total_returns }} {{ $project->total_returns > 1 ? 'returns' : 'return' }}</span>
                                    </td>
                                    <td>
                                        <span class="badge badge-info">{{ $project->material_requests_count }} MRs</span>
                                    </td>
                                    <td>{{ $project->latest_return_date ? \Carbon\Carbon::parse($project->latest_return_date)->format('d-M-Y') : 'N/A' }}</td>
                                    <td>
                                        <span class="badge badge-{{
                                            $project->primary_status === 'completed' ? 'success' :
                                            ($project->primary_status === 'pending_processing' ? 'warning' :
                                            ($project->primary_status === 'processing' ? 'info' :
                                            ($project->primary_status === 'rejected' ? 'danger' : 'secondary')))
                                        }}">
                                            {{ ucfirst(str_replace('_', ' ', $project->primary_status)) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-light">{{ number_format($project->total_items_returned) }} items</span>
                                    </td>
                                    <td>
                                        <div class="text-right">
                                            <div class="font-weight-medium">AED {{ number_format($project->total_value_returned ?? 0, 2) }}</div>
                                        </div>
                                    </td>
                                    <td style="white-space: nowrap;">
                                        <div class="d-flex align-items-center">
                                            @if($project->project_id == 0)
                                                <!-- View Button -->
                                                <a href="{{ route('warehouse.site-return.index') }}?unassigned=1" class="btn btn-sm btn-info mr-1" title="View Unassigned Returns">
                                                    <i class="material-icons mr-1" style="font-size: 14px;">visibility</i>
                                                    View
                                                </a>
                                                <!-- Print Button -->
                                                <button onclick="printUnassignedReturns()" class="btn btn-sm btn-outline-secondary mr-1" title="Print Unassigned Returns">
                                                    <i class="material-icons mr-1" style="font-size: 14px;">print</i>
                                                    Print
                                                </button>
                                                <!-- Export Button -->
                                                <button onclick="exportUnassignedReturns()" class="btn btn-sm btn-outline-info" title="Export Unassigned Returns">
                                                    <i class="material-icons" style="font-size: 14px;">file_download</i>
                                                </button>
                                            @else
                                                <!-- View Button -->
                                                <a href="{{ route('warehouse.site-return.project-returns', $project->project_id) }}" class="btn btn-sm btn-info mr-1" title="View Project Returns">
                                                    <i class="material-icons mr-1" style="font-size: 14px;">visibility</i>
                                                    View
                                                </a>
                                                <!-- Print Button -->
                                                <button onclick="printProjectReturns({{ $project->project_id }})" class="btn btn-sm btn-outline-secondary mr-1" title="Print Project Returns">
                                                    <i class="material-icons mr-1" style="font-size: 14px;">print</i>
                                                    Print
                                                </button>
                                                <!-- Export Button -->
                                                <button onclick="exportProjectReturns({{ $project->project_id }})" class="btn btn-sm btn-outline-info" title="Export Project Returns">
                                                    <i class="material-icons" style="font-size: 14px;">file_download</i>
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                    </div>
                </div>

                <!-- Pagination would go here if implemented -->
            @else
                <div class="text-center py-5">
                    <i class="material-icons text-muted mb-3" style="font-size: 64px;">assignment_return</i>
                    <h5 class="text-muted">No Projects with Returns Found</h5>
                    <p class="text-muted">No projects have site returns matching your current filters</p>
                    <a href="{{ route('warehouse.site-return.create') }}" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 18px;">add</i>
                        Create First Return
                    </a>
                </div>
            @endif
        </div>
    </div>

    <!-- Quick Statistics -->
    <div class="row mt-4">
        <div class="col-lg-6">
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="material-icons mr-2">trending_up</i>
                        Recent Activity
                    </h6>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        <div class="list-group-item d-flex align-items-center px-0">
                            <div class="mr-3">
                                <div class="bg-success text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                    <i class="material-icons" style="font-size: 16px;">check</i>
                                </div>
                            </div>
                            <div class="flex-grow-1">
                                <h6 class="mb-0">Today's Processing</h6>
                                <small class="text-muted">{{ $statistics['processed_today'] ?? 0 }} returns processed</small>
                            </div>
                        </div>
                        <div class="list-group-item d-flex align-items-center px-0">
                            <div class="mr-3">
                                <div class="bg-warning text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                    <i class="material-icons" style="font-size: 16px;">schedule</i>
                                </div>
                            </div>
                            <div class="flex-grow-1">
                                <h6 class="mb-0">Pending Processing</h6>
                                <small class="text-muted">{{ $statistics['pending_processing'] ?? 0 }} awaiting processing</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-6">
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="material-icons mr-2">work</i>
                        Top Return Reasons
                    </h6>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        @if(isset($topReturnReasons))
                            @foreach($topReturnReasons->take(3) as $reason => $count)
                                <div class="list-group-item d-flex align-items-center px-0">
                                    <div class="mr-3">
                                        <div class="bg-info text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                            <span style="font-size: 12px; font-weight: bold;">{{ $count }}</span>
                                        </div>
                                    </div>
                                    <div class="flex-grow-1">
                                        <h6 class="mb-0">{{ ucwords(str_replace('_', ' ', $reason)) }}</h6>
                                        <small class="text-muted">{{ $count }} returns</small>
                                    </div>
                                </div>
                            @endforeach
                        @else
                            <div class="text-center py-3">
                                <small class="text-muted">No return data available</small>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="{{ asset('admin-assets/css/warehouse-reports.css') }}">
@endpush

@push('scripts')
    <script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
<script>
        $(document).ready(function() {
            // Set filter values from URL parameters on page load
            const urlParams = new URLSearchParams(window.location.search);
            if (urlParams.get('project_id')) {
                $('#filter-project').val(urlParams.get('project_id')).trigger('change');
            }
            if (urlParams.get('division')) {
                $('#filter-division').val(urlParams.get('division')).trigger('change');
            }
            if (urlParams.get('search')) {
                $('#filter-search').val(urlParams.get('search'));
            }
            if (urlParams.get('sort')) {
                $('#filter-sort').val(urlParams.get('sort')).trigger('change');
            }
        });

        function applyFilters() {
            const projectId = $('#filter-project').val();
            const division = $('#filter-division').val();
            const search = $('#filter-search').val();
            const sort = $('#filter-sort').val();

            // Build query string
            const params = new URLSearchParams();
            if (projectId) params.set('project_id', projectId);
            if (division) params.set('division', division);
            if (search) params.set('search', search);
            if (sort) params.set('sort', sort);

            // Reload page with filters
            const queryString = params.toString();
            window.location.href = '{{ route("warehouse.site-return.index") }}' + (queryString ? '?' + queryString : '');
        }

        function clearFilters() {
            $('#filter-project').val('').trigger('change');
            $('#filter-division').val('').trigger('change');
            $('#filter-search').val('');
            $('#filter-sort').val('latest_return').trigger('change');

            // Reload page without filters
            window.location.href = '{{ route("warehouse.site-return.index") }}';
        }

        function exportReturns() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                alert('Export functionality will be implemented');
            }, 1000);
        }

        function printReturnNote(returnId) {
            window.open(`{{ url('warehouse/incoming-operations') }}/${returnId}/print`, '_blank');
        }

        function printMRReturns(materialRequestId) {
            alert('Print MR returns functionality will be implemented');
        }

        function printProjectReturns(projectId) {
            console.log('printProjectReturns called with project id:', projectId);

            // Open consolidated print view in new window
            const printUrl = '{{ route("warehouse.site-return.print-project-consolidated", ":id") }}'.replace(':id', projectId);
            console.log('Opening consolidated print URL:', printUrl);
            const printWindow = window.open(printUrl, '_blank', 'width=900,height=700,scrollbars=yes,resizable=yes');

            if (printWindow) {
                printWindow.focus();
            } else {
                // Fallback: navigate to print page if popup blocked
                window.open(printUrl, '_blank');
            }
        }

        function printUnassignedReturns() {
            console.log('printUnassignedReturns called');

            // Open consolidated print view for unassigned returns
            const printUrl = '{{ route("warehouse.site-return.print-project-consolidated", 0) }}';
            console.log('Opening unassigned returns print URL:', printUrl);
            const printWindow = window.open(printUrl, '_blank', 'width=900,height=700,scrollbars=yes,resizable=yes');

            if (printWindow) {
                printWindow.focus();
            } else {
                // Fallback: navigate to print page if popup blocked
                window.open(printUrl, '_blank');
            }
        }

        function exportProjectReturns(projectId) {
            alert('Export project returns functionality will be implemented');
        }

        function exportUnassignedReturns() {
            alert('Export unassigned returns functionality will be implemented');
        }

        // Auto-refresh every 5 minutes
        setInterval(() => {
            window.location.reload();
        }, 300000);
    </script>
@endpush