@extends('layouts.admin-simple')

@section('title', 'Reference Allocations - ' . $item->item_description)

@push('styles')
    <link href="{{ asset('admin-assets/css/warehouse-reports.css') }}" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
@endpush

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item">Reports</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.stock-on-project') }}">Stock on Project</a></li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.stock-on-project.details', $project->id) }}">{{ $project->project_name }}</a></li>
    <li class="breadcrumb-item active">Reference Allocations</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Reference Allocations</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">
                Item: {{ $item->item_description }} ({{ $item->item_code }}) |
                Project: {{ $project->project_name }}
            </p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <a href="{{ route('warehouse.reports.stock-on-project.details', $project->id) }}" class="btn btn-sm btn-secondary mr-2">
                <i class="material-icons mr-1" style="font-size: 16px;">arrow_back</i>
                Back to Stock Details
            </a>
            <button type="button" onclick="exportAllocations()" class="btn btn-sm btn-success mr-2" title="Export to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>
            <button type="button" onclick="printAllocations()" class="btn btn-sm btn-info" title="Print Report">
                <i class="material-icons mr-1" style="font-size: 16px;">print</i>
                Print
            </button>
        </div>
    </div>
@endsection

@section('content')
<div class="container-fluid">
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total References"
                :value="$references->count()"
                icon="receipt_long"
                color="primary"
                subtitle="Reference allocations"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Released Quantity"
                :value="number_format($totals['total_released'], 2)"
                icon="output"
                color="info"
                subtitle="Total quantity released"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Net Available"
                :value="number_format($totals['total_net'], 2)"
                icon="check_circle"
                color="success"
                subtitle="Net available quantity"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Value"
                :value="'AED ' . number_format($totals['total_value'], 2)"
                icon="attach_money"
                color="warning"
                subtitle="Total allocation value"
            />
        </div>
    </div>

    <!-- Item Information Card -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h6 class="mb-0">
                <i class="material-icons mr-2">inventory_2</i>
                Item Information
            </h6>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Item Code:</strong>
                        <span class="badge badge-light">{{ $item->item_code }}</span>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Description:</strong>
                        <span>{{ $item->item_description }}</span>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Unit:</strong>
                        <span class="badge badge-secondary">{{ $item->unit_of_measure }}</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4 warehouse-filters-panel">
        <div class="card-header">
            <h5 class="mb-0">
                <i class="material-icons align-middle mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form id="filterForm" class="warehouse-filter-form" method="GET" action="{{ route('warehouse.reports.item-reference-allocations', [$project->id, $item->id]) }}">
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label class="form-label">Reference Type</label>
                        <select name="type" class="form-control warehouse-select2">
                            <option value="">All Types</option>
                            <option value="Material Request" {{ request('type') == 'Material Request' ? 'selected' : '' }}>Material Request</option>
                            <option value="Direct Delivery" {{ request('type') == 'Direct Delivery' ? 'selected' : '' }}>Direct Delivery</option>
                            <option value="Material Transfer" {{ request('type') == 'Material Transfer' ? 'selected' : '' }}>Material Transfer</option>
                        </select>
                    </div>

                    <div class="col-md-3 mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-control warehouse-select2">
                            <option value="">All Statuses</option>
                            <option value="successful" {{ request('status') == 'successful' ? 'selected' : '' }}>Successful</option>
                            <option value="delivered" {{ request('status') == 'delivered' ? 'selected' : '' }}>Delivered</option>
                            <option value="approved" {{ request('status') == 'approved' ? 'selected' : '' }}>Approved</option>
                            <option value="received" {{ request('status') == 'received' ? 'selected' : '' }}>Received</option>
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date From</label>
                        <input type="date" name="date_from" value="{{ request('date_from') }}" class="form-control">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date To</label>
                        <input type="date" name="date_to" value="{{ request('date_to') }}" class="form-control">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary btn-block">
                            <i class="material-icons" style="font-size: 18px; vertical-align: middle;">search</i>
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Reference Allocations Table -->
    <div class="card warehouse-report-card">
        <div class="card-header">
            <h6 class="card-title mb-0">
                <i class="material-icons mr-2">receipt_long</i>
                Reference Allocations Detail
                @if($references->count() > 0)
                    <span class="badge badge-primary ml-2">{{ $references->count() }} References</span>
                @endif
            </h6>
        </div>
        <div class="card-body">
            @if($references->count() > 0)
                <div class="warehouse-table-container">
                    <div class="warehouse-loading-overlay" id="tableLoadingOverlay" style="display: none;">
                        <div class="warehouse-loading-content">
                            <div class="warehouse-loading-spinner"></div>
                            <div class="warehouse-loading-text">Loading reference allocations...</div>
                        </div>
                    </div>
                    <div class="warehouse-table-wrapper">
                        <table class="table table-hover mb-0 warehouse-table">
                            <thead class="warehouse-table-header">
                                <tr>
                                    <th class="warehouse-cell-nowrap">Type</th>
                                    <th class="warehouse-cell-nowrap">Reference #</th>
                                    <th class="warehouse-cell-date">Date</th>
                                    <th class="warehouse-cell-text">Requested By / Supplier</th>
                                    <th class="warehouse-cell-text">Division / From</th>
                                    <th class="warehouse-cell-status">Status</th>
                                    <th class="warehouse-cell-number text-right">Released</th>
                                    <th class="warehouse-cell-number text-right">Returned</th>
                                    <th class="warehouse-cell-number text-right">Transferred</th>
                                    <th class="warehouse-cell-number text-right">Net Qty</th>
                                    <th class="warehouse-cell-currency text-right">Value (AED)</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($references as $ref)
                                    @php
                                        $typeClass = $ref['type'] === 'Material Request' ? 'badge-primary' :
                                                    ($ref['type'] === 'Direct Delivery' ? 'badge-success' : 'badge-info');
                                        $statusClass = in_array(strtolower($ref['status']), ['successful', 'delivered', 'approved', 'received']) ?
                                                       'badge-success' : 'badge-warning';
                                    @endphp
                                    <tr class="warehouse-table-row">
                                        <td class="warehouse-cell-nowrap">
                                            <span class="badge {{ $typeClass }}">{{ $ref['type'] }}</span>
                                        </td>
                                        <td class="warehouse-cell-nowrap">
                                            <strong>{{ $ref['reference_number'] }}</strong>
                                        </td>
                                        <td class="warehouse-cell-date">{{ $ref['date'] }}</td>
                                        <td class="warehouse-cell-text">{{ $ref['requested_by'] }}</td>
                                        <td class="warehouse-cell-text">{{ $ref['division'] }}</td>
                                        <td class="warehouse-cell-status">
                                            <span class="badge {{ $statusClass }}">{{ $ref['status'] }}</span>
                                        </td>
                                        <td class="warehouse-cell-number text-right">
                                            {{ number_format($ref['quantity_released'], 2) }}
                                        </td>
                                        <td class="warehouse-cell-number text-right text-warning">
                                            {{ number_format($ref['quantity_returned'], 2) }}
                                        </td>
                                        <td class="warehouse-cell-number text-right text-info">
                                            {{ number_format($ref['quantity_transferred'], 2) }}
                                        </td>
                                        <td class="warehouse-cell-number text-right text-success">
                                            <strong>{{ number_format($ref['net_quantity'], 2) }}</strong>
                                        </td>
                                        <td class="warehouse-cell-currency text-right">
                                            AED {{ number_format($ref['total_value'], 2) }}
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                            <tfoot class="warehouse-table-footer">
                                <tr>
                                    <th colspan="6" class="warehouse-cell-total"><strong>Total</strong></th>
                                    <th class="warehouse-cell-number text-right">
                                        <strong>{{ number_format($totals['total_released'], 2) }}</strong>
                                    </th>
                                    <th class="warehouse-cell-number text-right text-warning">
                                        <strong>{{ number_format($totals['total_returned'], 2) }}</strong>
                                    </th>
                                    <th class="warehouse-cell-number text-right text-info">
                                        <strong>{{ number_format($totals['total_transferred'], 2) }}</strong>
                                    </th>
                                    <th class="warehouse-cell-number text-right text-success">
                                        <strong>{{ number_format($totals['total_net'], 2) }}</strong>
                                    </th>
                                    <th class="warehouse-cell-currency text-right">
                                        <strong>AED {{ number_format($totals['total_value'], 2) }}</strong>
                                    </th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            @else
                <div class="warehouse-empty-state">
                    <div class="warehouse-empty-icon">
                        <i class="material-icons">inbox</i>
                    </div>
                    <h5 class="warehouse-empty-title">No Reference Allocations Found</h5>
                    <p class="warehouse-empty-description">
                        No reference allocations found for this item in the selected project.<br>
                        This means the item hasn't been delivered or transferred to this project.
                    </p>
                    <div class="warehouse-empty-actions">
                        <a href="{{ route('warehouse.reports.stock-on-project.details', $project->id) }}" class="btn btn-primary">
                            <i class="material-icons mr-1">arrow_back</i>
                            Back to Stock Details
                        </a>
                    </div>
                </div>
            @endif
        </div>
    </div>

    <!-- Additional Information -->
    @if($references->count() > 0)
    <div class="card mt-4">
        <div class="card-header">
            <h6 class="card-title mb-0">
                <i class="material-icons mr-2">info</i>
                Allocation Summary
            </h6>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Reference Type Distribution</h6>
                    @php
                        $materialRequests = $references->where('type', 'Material Request')->count();
                        $directDeliveries = $references->where('type', 'Direct Delivery')->count();
                        $materialTransfers = $references->where('type', 'Material Transfer')->count();
                    @endphp
                    <div class="mb-2">
                        <span class="badge badge-primary mr-2">Material Requests: {{ $materialRequests }}</span>
                    </div>
                    <div class="mb-2">
                        <span class="badge badge-success mr-2">Direct Deliveries: {{ $directDeliveries }}</span>
                    </div>
                    <div class="mb-2">
                        <span class="badge badge-info mr-2">Material Transfers: {{ $materialTransfers }}</span>
                    </div>
                </div>
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Key Metrics</h6>
                    <div class="mb-2">
                        <strong>Total References:</strong> {{ $references->count() }}
                    </div>
                    <div class="mb-2">
                        <strong>Total Value:</strong> AED {{ number_format($totals['total_value'], 2) }}
                    </div>
                    <div class="mb-2">
                        <strong>Return Rate:</strong>
                        @if($totals['total_released'] > 0)
                            {{ number_format(($totals['total_returned'] / $totals['total_released']) * 100, 2) }}%
                        @else
                            0%
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif
</div>
@endsection

@push('scripts')
    <script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize warehouse table functionality
            if (typeof initializeWarehouseTable === 'function') {
                initializeWarehouseTable();
            }
        });

        function exportAllocations() {
            showLoadingOverlay();

            // Simulate export process
            setTimeout(function() {
                hideLoadingOverlay();
                alert('Export functionality will be implemented soon.');
            }, 1000);
        }

        function printAllocations() {
            window.print();
        }

        function showLoadingOverlay() {
            $('#tableLoadingOverlay').show();
        }

        function hideLoadingOverlay() {
            $('#tableLoadingOverlay').hide();
        }
    </script>
@endpush