@extends('layouts.admin-simple')

@section('title', 'Damaged Items Details - ' . ($item->item_description ?? 'Unknown Item'))

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.damaged-items') }}">Damaged Items</a></li>
    <li class="breadcrumb-item active">{{ $item->item_code ?? 'Details' }}</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Damaged Items Details</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">
                Item: {{ $item->item_description ?? 'Unknown Item' }} |
                Code: {{ $item->item_code ?? 'No Code' }} |
                Total Damaged: {{ $damagedItems->count() }} instances
            </p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <a href="{{ route('warehouse.reports.damaged-items') }}{{ request()->getQueryString() ? '?' . request()->getQueryString() : '' }}"
               class="btn btn-sm btn-secondary mr-2">
                <i class="material-icons mr-1" style="font-size: 16px;">arrow_back</i>
                Back to Summary
            </a>
            <button type="button" onclick="exportDamagedItems()" class="btn btn-sm btn-success mr-2" title="Export to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>
            <button type="button" onclick="window.print()" class="btn btn-sm btn-info" title="Print Report">
                <i class="material-icons mr-1" style="font-size: 16px;">print</i>
                Print
            </button>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Instances"
                :value="$damagedItems->total()"
                icon="warning"
                color="danger"
                subtitle="Total damaged instances"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Quantity"
                :value="number_format($totalQuantity, 2)"
                icon="format_list_numbered"
                color="warning"
                subtitle="Total damaged quantity"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Cost"
                :value="number_format($totalCost, 2)"
                :currency="true"
                icon="attach_money"
                color="danger"
                subtitle="Total financial impact"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Net Loss"
                :value="number_format($totalLoss, 2)"
                :currency="true"
                icon="trending_down"
                color="danger"
                subtitle="Total unrecoverable loss"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="FOC Available"
                :value="number_format($totalFocAvailable, 2)"
                icon="recycling"
                color="success"
                subtitle="Current FOC inventory"
            />
        </div>
    </div>

    <!-- FOC Lifecycle Statistics -->
    <div class="row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total FOC Received"
                :value="number_format($totalFocReceived, 2)"
                icon="input"
                color="info"
                subtitle="Total converted to FOC"
            />
        </div>
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="FOC Dispatched"
                :value="number_format($totalFocDispatched, 2)"
                icon="output"
                color="warning"
                subtitle="Total issued from FOC"
            />
        </div>
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="FOC Returned"
                :value="number_format($totalFocReturned, 2)"
                icon="keyboard_return"
                color="primary"
                subtitle="Returned to FOC inventory"
            />
        </div>
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Net FOC Position"
                :value="number_format($totalFocReceived - $totalFocDispatched + $totalFocReturned, 2)"
                icon="account_balance"
                color="secondary"
                subtitle="Expected available balance"
            />
        </div>
    </div>

    <!-- Item Information Card -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h6 class="mb-0">
                <i class="material-icons mr-2">inventory_2</i>
                Item Information
            </h6>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Item Code:</strong>
                        <span class="badge badge-light">{{ $item->item_code ?? 'No Code' }}</span>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Description:</strong>
                        <span>{{ $item->item_description ?? 'Unknown Item' }}</span>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="d-flex align-items-center mb-2">
                        <strong class="mr-2">Category:</strong>
                        <span class="badge badge-secondary">{{ $item->category->category_name ?? 'No Category' }}</span>
                    </div>
                </div>
            </div>
        </div>
    </div>


    <!-- Filters Panel -->
    <div class="card mb-4 warehouse-filters-panel">
        <div class="card-header">
            <h5 class="mb-0">
                <i class="material-icons align-middle mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form id="filterForm" class="warehouse-filter-form" method="GET" action="{{ route('warehouse.reports.damaged-items-details', $item->id) }}">
                <div class="row">
                    <div class="col-md-2 mb-3">
                        <label class="form-label">Source Type</label>
                        <select name="source_type" class="form-control warehouse-select2">
                            <option value="">All Sources</option>
                            @foreach($sourceTypes as $key => $value)
                                <option value="{{ $key }}" {{ request('source_type') == $key ? 'selected' : '' }}>
                                    {{ $value }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-control warehouse-select2">
                            <option value="">All Statuses</option>
                            @foreach($statuses as $key => $value)
                                <option value="{{ $key }}" {{ request('status') == $key ? 'selected' : '' }}>
                                    {{ $value }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Severity</label>
                        <select name="severity" class="form-control warehouse-select2">
                            <option value="">All Severities</option>
                            @foreach($severities as $key => $value)
                                <option value="{{ $key }}" {{ request('severity') == $key ? 'selected' : '' }}>
                                    {{ $value }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Responsibility</label>
                        <select name="cost_responsibility" class="form-control warehouse-select2">
                            <option value="">All Responsibilities</option>
                            @foreach($responsibilities as $key => $value)
                                <option value="{{ $key }}" {{ request('cost_responsibility') == $key ? 'selected' : '' }}>
                                    {{ $value }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Project</label>
                        <select name="project_id" class="form-control warehouse-select2">
                            <option value="">All Projects</option>
                            @foreach($projects as $project)
                                <option value="{{ $project->id }}" {{ request('project_id') == $project->id ? 'selected' : '' }}>
                                    {{ $project->project_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary btn-block">
                            <i class="material-icons" style="font-size: 18px; vertical-align: middle;">search</i>
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>


    <!-- FOC Inventory Section -->
    @if($focInventory->count() > 0)
    <div class="card mb-3">
        <div class="card-header">
            <h5 class="mb-0">
                <i class="material-icons align-middle mr-2">recycling</i>
                FOC Inventory Available ({{ number_format($totalFocAvailable, 2) }} units)
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                @foreach($focInventory as $foc)
                <div class="col-md-3 mb-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <strong>{{ ucfirst($foc->condition) }}</strong>
                            <small class="text-muted d-block">{{ $foc->storage_location ?? 'No location' }}</small>
                        </div>
                        <span class="badge badge-success">
                            {{ number_format($foc->quantity_available, 2) }}
                        </span>
                    </div>
                </div>
                @endforeach
            </div>
        </div>
    </div>
    @endif

    <!-- Damaged Items Details Table -->
    <div class="card">
        <div class="card-header warehouse-report-header">
            <h5 class="mb-0 warehouse-report-title">
                <i class="material-icons align-middle mr-2">warning</i>
                Damage Records for {{ $item->item_code }} ({{ $damagedItems->total() }} records)
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="warehouse-table-container">
                <!-- Loading Overlay -->
                <div class="warehouse-loading-overlay" id="tableLoadingOverlay">
                    <div class="warehouse-loading-content">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="sr-only">Loading...</span>
                            </div>
                            <div class="mt-2">Loading damaged items data...</div>
                        </div>
                    </div>
                </div>

                <!-- Table -->
                <div class="warehouse-table-wrapper">
                    <table class="table table-hover mb-0 warehouse-table">
                        <thead class="thead-light">
                            <tr>
                                <th style="min-width: 100px;">Date</th>
                                <th style="min-width: 140px;">Source Reference</th>
                                <th style="min-width: 80px;" class="text-right">Quantity</th>
                                <th style="min-width: 100px;" class="text-center">Condition</th>
                                <th style="min-width: 100px;" class="text-center">Severity</th>
                                <th style="min-width: 100px;" class="text-center">Status</th>
                                <th style="min-width: 100px;" class="text-right">Total Cost</th>
                                <th style="min-width: 100px;" class="text-right">Recoverable</th>
                                <th style="min-width: 140px;" class="text-center">Project</th>
                                <th style="min-width: 120px;" class="text-center">Responsibility</th>
                                <th style="min-width: 200px;">Description</th>
                                <th style="min-width: 80px;" class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($damagedItems as $damagedItem)
                                <tr>
                                    <td class="warehouse-cell-date">
                                        {{ $damagedItem->created_at->format('d M Y') }}
                                        <br><small class="text-muted">{{ $damagedItem->created_at->format('H:i') }}</small>
                                    </td>
                                    <td class="warehouse-cell-nowrap">
                                        <strong>{{ $damagedItem->source_reference ?? 'No reference' }}</strong>
                                        <br><small class="text-muted">{{ $sourceTypes[$damagedItem->source_type] ?? $damagedItem->source_type }}</small>
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        <strong>{{ number_format($damagedItem->quantity, 2) }}</strong>
                                        <br><small class="text-muted">{{ $damagedItem->unit }}</small>
                                    </td>
                                    <td class="warehouse-cell-status">
                                        <span class="badge warehouse-badge warehouse-status-{{
                                            $damagedItem->condition === 'damaged' ? 'danger' :
                                            ($damagedItem->condition === 'excess' ? 'warning' : 'info')
                                        }}">
                                            {{ ucfirst($damagedItem->condition) }}
                                        </span>
                                    </td>
                                    <td class="warehouse-cell-status">
                                        <span class="badge warehouse-badge warehouse-status-{{
                                            $damagedItem->damage_severity === 'severe' || $damagedItem->damage_severity === 'major' ? 'danger' :
                                            ($damagedItem->damage_severity === 'moderate' ? 'warning' : 'secondary')
                                        }}">
                                            {{ $severities[$damagedItem->damage_severity] ?? ucfirst($damagedItem->damage_severity) }}
                                        </span>
                                    </td>
                                    <td class="warehouse-cell-status">
                                        <span class="badge warehouse-badge warehouse-status-{{
                                            $damagedItem->status === 'reported' ? 'warning' :
                                            ($damagedItem->status === 'investigating' ? 'info' :
                                            ($damagedItem->status === 'approved' ? 'primary' :
                                            ($damagedItem->status === 'settled' ? 'success' : 'danger')))
                                        }}">
                                            {{ $statuses[$damagedItem->status] ?? ucfirst($damagedItem->status) }}
                                        </span>
                                    </td>
                                    <td class="text-right warehouse-cell-currency">
                                        <div class="d-flex align-items-center justify-content-end">
                                            <span class="dirham-symbol text-dark mr-1" style="width: 16px; height: 16px; display: inline-flex; align-items: center;">
                                                <svg viewBox="0 0 1000 870" width="16" height="16" style="fill: currentColor;">
                                                    <path d="m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.3 0.7 19.2 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z"></path>
                                                </svg>
                                            </span>
                                            <div class="font-weight-medium text-dark">{{ number_format($damagedItem->total_cost, 2) }}</div>
                                        </div>
                                    </td>
                                    <td class="text-right warehouse-cell-currency">
                                        <div class="d-flex align-items-center justify-content-end">
                                            <span class="dirham-symbol text-success mr-1" style="width: 16px; height: 16px; display: inline-flex; align-items: center;">
                                                <svg viewBox="0 0 1000 870" width="16" height="16" style="fill: currentColor;">
                                                    <path d="m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.3 0.7 19.2 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z"></path>
                                                </svg>
                                            </span>
                                            <div class="font-weight-medium text-success">{{ number_format($damagedItem->recoverable_amount, 2) }}</div>
                                        </div>
                                        <br><small class="text-muted">{{ number_format(($damagedItem->recoverable_amount / max($damagedItem->total_cost, 1)) * 100, 1) }}%</small>
                                    </td>
                                    <td class="text-center">
                                        <span class="badge warehouse-badge warehouse-status-info">
                                            {{ $damagedItem->project->project_name ?? 'No Project' }}
                                        </span>
                                    </td>
                                    <td class="text-center">
                                        <span class="badge warehouse-badge warehouse-status-{{
                                            $damagedItem->cost_responsibility === 'company' ? 'danger' :
                                            ($damagedItem->cost_responsibility === 'supplier' ? 'warning' : 'secondary')
                                        }}">
                                            {{ $responsibilities[$damagedItem->cost_responsibility] ?? ucfirst($damagedItem->cost_responsibility) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="text-truncate" style="max-width: 200px;" title="{{ $damagedItem->damage_description }}">
                                            {{ Str::limit($damagedItem->damage_description ?? 'No description', 50) }}
                                        </div>
                                    </td>
                                    <td class="text-center">
                                        <button class="btn btn-sm btn-outline-primary warehouse-action-btn"
                                                onclick="viewDamagedItemDetails({{ $damagedItem->id }})"
                                                title="View Full Details">
                                            <i class="material-icons">visibility</i>
                                        </button>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="12" class="warehouse-empty-state">
                                        <i class="material-icons">warning</i>
                                        <h5>No Damage Records Found</h5>
                                        <p class="text-muted">No damage records found for this item with current filters.</p>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Pagination -->
    @if(method_exists($damagedItems, 'links'))
        <div class="d-flex justify-content-center mt-3">
            {{ $damagedItems->links() }}
        </div>
    @endif
@endsection

<!-- Damaged Item Details Modal -->
<div class="modal fade" id="damagedItemModal" tabindex="-1" role="dialog" aria-labelledby="damagedItemModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="damagedItemModalLabel">
                    <i class="material-icons align-middle mr-2">warning</i>
                    Damaged Item Details
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="damagedItemContent">
                    <!-- Content will be loaded here -->
                    <div class="text-center py-4">
                        <div class="spinner-border text-primary" role="status">
                            <span class="sr-only">Loading...</span>
                        </div>
                        <div class="mt-2">Loading details...</div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

@push('styles')
    <link href="{{ asset('admin-assets/css/warehouse-reports.css') }}" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
@endpush

@push('scripts')
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
<script>
function exportDamagedItems() {
    const currentUrl = new URL(window.location);
    currentUrl.searchParams.set('export', 'excel');
    window.open(currentUrl.toString(), '_blank');
}

function viewDamagedItemDetails(damagedItemId) {
    $('#damagedItemModal').modal('show');

    // Reset content to loading state
    $('#damagedItemContent').html(`
        <div class="text-center py-4">
            <div class="spinner-border text-primary" role="status">
                <span class="sr-only">Loading...</span>
            </div>
            <div class="mt-2">Loading details...</div>
        </div>
    `);

    // Load damaged item details via AJAX
    $.ajax({
        url: `{{ url('warehouse/reports/damaged-items/item') }}/${damagedItemId}/details`,
        method: 'GET',
        success: function(data) {
            $('#damagedItemContent').html(data);
        },
        error: function(xhr) {
            $('#damagedItemContent').html(`
                <div class="alert alert-danger">
                    <strong>Error:</strong> Failed to load damaged item details.
                    <br>Please try again or contact support if the problem persists.
                </div>
            `);
        }
    });
}

// Initialize Select2 for better dropdown experience
$(document).ready(function() {
    $('.warehouse-select2').select2({
        theme: 'bootstrap4',
        width: '100%'
    });
});
</script>
@endpush