@extends('layouts.admin-simple')

@section('title', 'Item Categories')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Item Categories</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Item Categories</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage item categories and types of goods</p>
        </div>
        @can('warehouse.item-categories.create')
        <div class="page-actions-right d-flex align-items-center">
            <button onclick="showImportModal()" class="btn btn-sm btn-info mr-2" title="Import item categories">
                <i class="material-icons mr-1" style="font-size: 16px;">file_upload</i>
                Import
            </button>
            <button onclick="showAddCategoryModal()" class="btn btn-sm btn-primary" title="Add new item category">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                Add Category
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Filters Row -->
    <div class="card mb-3">
        <div class="card-body py-2">
            <form method="GET" action="{{ route('warehouse.item-categories.index') }}" id="filterForm">
                <div class="row align-items-center">
                    <div class="col-md-3">
                        <div class="form-group mb-0">
                            <label for="search" class="sr-only">Search</label>
                            <input type="text" name="search" id="search" class="form-control form-control-sm"
                                   placeholder="Search categories..." value="{{ request('search') }}">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group mb-0">
                            <label for="status" class="sr-only">Status</label>
                            <select name="status" id="status" class="form-control form-control-sm">
                                <option value="">All Statuses</option>
                                <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group mb-0">
                            <label for="type_of_goods" class="sr-only">Type of Goods</label>
                            <select name="type_of_goods" id="type_of_goods_filter" class="form-control form-control-sm">
                                <option value="">All Types</option>
                                <option value="Raw Material" {{ request('type_of_goods') == 'Raw Material' ? 'selected' : '' }}>Raw Material</option>
                                <option value="Finished Goods" {{ request('type_of_goods') == 'Finished Goods' ? 'selected' : '' }}>Finished Goods</option>
                                <option value="Work in Progress" {{ request('type_of_goods') == 'Work in Progress' ? 'selected' : '' }}>Work in Progress</option>
                                <option value="Consumables" {{ request('type_of_goods') == 'Consumables' ? 'selected' : '' }}>Consumables</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group mb-0">
                            <label for="per_page" class="sr-only">Per Page</label>
                            <select name="per_page" id="per_page" class="form-control form-control-sm">
                                <option value="10" {{ request('per_page') == 10 ? 'selected' : '' }}>10 per page</option>
                                <option value="25" {{ request('per_page', 25) == 25 ? 'selected' : '' }}>25 per page</option>
                                <option value="50" {{ request('per_page') == 50 ? 'selected' : '' }}>50 per page</option>
                                <option value="100" {{ request('per_page') == 100 ? 'selected' : '' }}>100 per page</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2 text-right">
                        <button type="submit" class="btn btn-sm btn-secondary mr-2">
                            <i class="material-icons mr-1" style="font-size: 14px;">filter_list</i>Apply
                        </button>
                        <a href="{{ route('warehouse.item-categories.index') }}" class="btn btn-sm btn-outline-secondary">
                            <i class="material-icons mr-1" style="font-size: 14px;">clear</i>Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Categories Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">category</i>
                Item Categories List
                <span class="badge badge-secondary ml-2">{{ $categories->total() }} Total</span>
            </h5>
            <div class="text-muted" style="font-size: 14px;">
                Showing {{ $categories->firstItem() ?? 0 }} - {{ $categories->lastItem() ?? 0 }} of {{ $categories->total() }}
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Category Info</th>
                            <th>Type of Goods</th>
                            <th>Status</th>
                            <th>Items Count</th>
                            <th>Created Date</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($categories as $category)
                        <tr>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <i class="material-icons" style="font-size: 20px;">category</i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">{{ $category->category_name }}</div>
                                        <div class="mb-1">
                                            <code class="category-code">{{ $category->category_code ?: 'N/A' }}</code>
                                        </div>
                                        <small class="text-muted">{{ $category->description ?: 'No description' }}</small>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="badge badge-info">{{ $category->type_of_goods }}</span>
                            </td>
                            <td>
                                @if($category->status === 'active')
                                    <span class="status-badge status-active">Active</span>
                                @else
                                    <span class="status-badge status-inactive">Inactive</span>
                                @endif
                            </td>
                            <td>
                                <span class="badge badge-secondary">{{ $category->active_items_count ?? 0 }} items</span>
                            </td>
                            <td>
                                <div>
                                    <div class="small">{{ $category->created_at->format('M d, Y H:i') }}</div>
                                    <div class="small text-muted">by Admin</div>
                                </div>
                            </td>
                            <td class="text-right">
                                <div class="btn-group" role="group">
                                    <a href="{{ route('warehouse.item-categories.show', $category->id) }}" class="btn btn-sm btn-outline-primary" title="View">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>
                                    @can('warehouse.item-categories.edit')
                                    <a href="{{ route('warehouse.item-categories.edit', $category->id) }}" class="btn btn-sm btn-outline-success" title="Edit">
                                        <i class="material-icons" style="font-size: 16px;">edit</i>
                                    </a>
                                    @endcan
                                    @can('warehouse.item-categories.delete')
                                    <button onclick="deleteCategory({{ $category->id }})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>
                                    @endcan
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="6" class="text-center py-4">
                                <i class="material-icons text-muted" style="font-size: 48px;">category</i>
                                <p class="text-muted mt-2">No item categories found.</p>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Laravel Pagination Links -->
            <div class="d-flex justify-content-between align-items-center mt-4">
                <div>
                    @if($categories->hasPages())
                        <p class="text-muted mb-0">
                            Showing page {{ $categories->currentPage() }} of {{ $categories->lastPage() }}
                        </p>
                    @endif
                </div>
                <div>
                    {{ $categories->links('pagination::bootstrap-4') }}
                </div>
            </div>
        </div>
    </div>
@endsection

<!-- Add Category Modal -->
<div class="modal fade" id="addCategoryModal" tabindex="-1" role="dialog" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addCategoryModalLabel">
                    <i class="material-icons mr-2">category</i>
                    Add New Item Category
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addCategoryForm">
                    @csrf
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="category_name">Category Name <span class="text-danger">*</span></label>
                                <input type="text" name="category_name" id="category_name" class="form-control" required>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="category_code">Category Code</label>
                                <input type="text" name="category_code" id="category_code" class="form-control" maxlength="20" placeholder="Auto-generated if empty">
                                <small class="form-text text-muted">Leave empty to auto-generate from category name</small>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <x-warehouse.select2-dropdown
                                name="type_of_goods"
                                id="type_of_goods"
                                label="Type of Goods"
                                placeholder="Loading types..."
                                :options="[]"
                                required="true"
                            />
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <x-warehouse.select2-dropdown
                                name="status"
                                id="status"
                                label="Status"
                                placeholder="Select Status"
                                :options="[
                                    ['value' => 'active', 'text' => 'Active'],
                                    ['value' => 'inactive', 'text' => 'Inactive']
                                ]"
                                selected="active"
                                required="true"
                            />
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="description">Description</label>
                                <textarea name="description" id="description" class="form-control" rows="3" placeholder="Optional description for this category"></textarea>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" onclick="saveCategory()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                    Save Category
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Import Categories Modal -->
<div class="modal fade" id="importModal" tabindex="-1" role="dialog" aria-labelledby="importModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="importModalLabel">
                    <i class="material-icons mr-2">file_upload</i>
                    Import Item Categories
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <!-- Step 1: Download Template -->
                <div class="card mb-3">
                    <div class="card-body py-3">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0">Step 1: Download Template</h6>
                            <button type="button" onclick="downloadTemplate()" class="btn btn-sm btn-outline-info">
                                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                                Download Template
                            </button>
                        </div>
                        <small class="text-muted">Download the Excel template with the correct column format before uploading your category data.</small>
                    </div>
                </div>

                <hr>

                <!-- Step 2: Upload File -->
                <form id="importForm" enctype="multipart/form-data">
                    @csrf
                    <div class="form-group">
                        <h6 class="mb-3">Step 2: Upload Your File</h6>
                        <div class="custom-file">
                            <input type="file" name="file" class="custom-file-input" id="importFile" accept=".xlsx,.xls,.csv" required>
                            <label class="custom-file-label" for="importFile">Choose file...</label>
                        </div>
                        <small class="form-text text-muted mt-2">
                            <i class="material-icons" style="font-size: 14px;">info</i>
                            Supported formats: Excel (.xlsx, .xls) or CSV files. Maximum size: 10MB.
                        </small>
                    </div>

                    <!-- Progress Bar -->
                    <div class="progress mb-3" id="uploadProgress" style="display: none;">
                        <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                    </div>

                    <!-- Import Results -->
                    <div id="importResults" style="display: none;">
                        <div class="alert alert-info">
                            <h6><i class="material-icons mr-1">assessment</i> Import Results</h6>
                            <div id="importSummary"></div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" onclick="importCategories()" class="btn btn-primary" id="importBtn">
                    <i class="material-icons mr-1" style="font-size: 16px;">cloud_upload</i>
                    Import Categories
                </button>
            </div>
        </div>
    </div>
</div>

@push('styles')
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-active {
            background-color: #d4edda;
            color: #155724;
        }

        .status-inactive {
            background-color: #f8d7da;
            color: #721c24;
        }

        /* Modal z-index fix - highest priority */
        .modal,
        #addCategoryModal,
        #importModal {
            z-index: 999999999 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        .modal-backdrop,
        .modal-backdrop.fade,
        .modal-backdrop.show {
            z-index: 999999998 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        /* Hide all potential overlapping elements */
        body.modal-open .navbar,
        body.modal-open .main-header,
        body.modal-open .sidebar,
        body.modal-open .main-sidebar,
        body.modal-open .content-wrapper,
        body.modal-open .overlay,
        body.modal-open .page-overlay,
        body.modal-open .loading-overlay,
        body.modal-open .nav,
        body.modal-open .navigation,
        body.modal-open .header,
        body.modal-open .footer,
        body.modal-open .breadcrumb,
        body.modal-open .topbar,
        body.modal-open .app-header {
            z-index: 1 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Force modal to appear above everything */
        .modal.show,
        .modal.fade.show {
            display: block !important;
            z-index: 999999 !important;
            opacity: 1 !important;
        }

        /* Override any potential conflicts */
        .modal * {
            position: relative;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        /* Fix Select2 dropdown z-index in modals */
        .modal .select2-container--default .select2-dropdown,
        .modal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        .modal .select2-container--default.select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
        }

        .modal .select2-container {
            z-index: 1000000000 !important;
            position: relative !important;
        }

        .modal .select2-container .select2-selection {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Specific fixes for category modals */
        #importModal .select2-dropdown,
        #importModal .select2-container--open .select2-dropdown,
        #addCategoryModal .select2-dropdown,
        #addCategoryModal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        /* Ensure Select2 dropdowns appear above modal backdrop */
        .select2-dropdown {
            z-index: 1000000002 !important;
        }

        /* Override Select2's default z-index when in modal context */
        .modal-open .select2-dropdown {
            z-index: 1000000002 !important;
        }

        .modal-open .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
        }

        /* Force hide any overlay elements that might interfere */
        body.modal-open *[class*="overlay"],
        body.modal-open *[id*="overlay"],
        body.modal-open .fixed-top:not(.modal):not(.modal-backdrop),
        body.modal-open .fixed-bottom:not(.modal):not(.modal-backdrop),
        body.modal-open .sticky-top:not(.modal):not(.modal-backdrop) {
            display: none !important;
        }

        /* Ensure modal is always on top */
        .modal {
            z-index: 999999999 !important;
        }

        .modal-backdrop {
            z-index: 999999998 !important;
        }

        /* Specific styling for Select2 dropdowns in modals */
        .modal-select2-dropdown {
            z-index: 1000000004 !important;
            position: absolute !important;
        }

        .modal-select2-container {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Fix positioning for Select2 dropdowns opened from modals */
        body > .select2-container--open .select2-dropdown.modal-select2-dropdown {
            z-index: 1000000004 !important;
        }

        /* Override default Select2 positioning for modal dropdowns */
        .select2-dropdown.modal-select2-dropdown {
            z-index: 1000000004 !important;
        }

        /* Force Select2 dropdowns to appear above modal */
        .select2-container--open .select2-dropdown--above,
        .select2-container--open .select2-dropdown--below {
            z-index: 1000000002 !important;
        }

        /* Ensure Select2 results are visible with highest priority */
        .select2-results {
            z-index: 1000000005 !important;
            position: relative !important;
        }

        .select2-results__options {
            z-index: 1000000005 !important;
            position: relative !important;
        }

        .select2-results__option {
            z-index: 1000000005 !important;
            position: relative !important;
        }

        /* Force all Select2 dropdown elements to highest z-index in modals */
        .modal .select2-dropdown,
        .modal-select2-dropdown,
        body .select2-container--open .select2-dropdown {
            z-index: 1000000004 !important;
            position: absolute !important;
        }

        .modal .select2-dropdown .select2-results,
        .modal-select2-dropdown .select2-results {
            z-index: 1000000005 !important;
            position: relative !important;
        }

        .modal .select2-dropdown .select2-results__options,
        .modal-select2-dropdown .select2-results__options {
            z-index: 1000000005 !important;
            position: relative !important;
        }

        /* Specific override for any conflicting styles */
        body .modal .select2-dropdown--below,
        body .modal .select2-dropdown--above {
            z-index: 1000000004 !important;
        }

        body .modal .select2-dropdown--below .select2-results,
        body .modal .select2-dropdown--above .select2-results {
            z-index: 1000000005 !important;
        }

        /* Ensure proper positioning calculation for modal Select2 dropdowns */
        .modal.show .select2-dropdown.modal-select2-dropdown {
            position: absolute !important;
            z-index: 1000000004 !important;
        }

        /* Fix width and positioning issues */
        .modal .select2-container.modal-select2-container {
            width: 100% !important;
        }

        .modal .select2-dropdown.modal-select2-dropdown {
            width: auto !important;
            min-width: 100% !important;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .category-code {
            font-size: 10px;
            font-weight: bold;
            color: #007bff;
            background-color: #e3f2fd;
            padding: 2px 6px;
            border-radius: 3px;
            text-transform: uppercase;
        }
    </style>
@endpush

@push('scripts')
    <script>

        $(document).ready(function() {
            // AGGRESSIVE fix for loading overlay - use !important to override inline styles
            function forceHideLoadingOverlay() {
                const overlay = $('#pageLoadingOverlay')[0];
                if (overlay) {
                    // Remove all classes and set style with !important
                    overlay.className = '';
                    overlay.style.cssText = 'display: none !important; visibility: hidden !important; opacity: 0 !important; pointer-events: none !important;';
                    console.log('🚀 FORCE HIDDEN: Loading overlay aggressively hidden');
                }
            }

            // Force hide immediately
            forceHideLoadingOverlay();

            // Set up mutation observer to watch for changes to the overlay
            const overlay = document.getElementById('pageLoadingOverlay');
            if (overlay) {
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'attributes' &&
                            (mutation.attributeName === 'style' || mutation.attributeName === 'class')) {
                            const element = mutation.target;
                            const isVisible = element.style.display !== 'none' ||
                                            element.classList.contains('show') ||
                                            window.getComputedStyle(element).display !== 'none';

                            if (isVisible) {
                                console.warn('🔄 Loading overlay tried to show, force hiding again');
                                forceHideLoadingOverlay();
                            }
                        }
                    });
                });

                observer.observe(overlay, {
                    attributes: true,
                    attributeFilter: ['style', 'class']
                });

                console.log('👀 Mutation observer set up to watch loading overlay');
            }

            // Additional check after a short delay to ensure overlay is hidden
            setTimeout(function() {
                if ($('#pageLoadingOverlay').is(':visible')) {
                    console.warn('🚨 Loading overlay still visible after 500ms, force hiding it');
                    forceHideLoadingOverlay();
                }
            }, 500);

            // Prevent dropdown events from interfering with modals
            $(document).on('click', function(e) {
                // If a modal is open, prevent dropdown events from interfering
                if ($('.modal.show').length > 0) {
                    // Allow clicks inside modals and dropdown elements
                    if ($(e.target).closest('.modal, .dropdown-menu, [data-toggle="dropdown"]').length > 0) {
                        return;
                    }
                    // Prevent other events from bubbling
                    e.stopPropagation();
                }
            });

            // Monitor Select2 open events in modals
            $(document).on('select2:open', function(e) {
                if ($(e.target).closest('.modal').length > 0) {
                    console.log('🔓 Select2 opened in modal, fixing positioning and z-index...');

                    // Force z-index and positioning on the dropdown when it opens
                    setTimeout(function() {
                        const $dropdown = $('.select2-dropdown');
                        const $select = $(e.target);

                        // Apply high z-index
                        $dropdown.css({
                            'z-index': '1000000004',
                            'position': 'absolute'
                        });
                        $('.select2-results').css({
                            'z-index': '1000000005',
                            'position': 'relative'
                        });
                        $('.select2-results__options').css({
                            'z-index': '1000000005',
                            'position': 'relative'
                        });
                        $('.select2-results__option').css({
                            'z-index': '1000000005',
                            'position': 'relative'
                        });

                        // Fix positioning if it seems off
                        const $container = $select.next('.select2-container');
                        if ($container.length && $dropdown.length) {
                            const containerOffset = $container.offset();
                            const containerHeight = $container.outerHeight();

                            // Position dropdown directly below the container
                            $dropdown.css({
                                'left': containerOffset.left + 'px',
                                'top': (containerOffset.top + containerHeight) + 'px',
                                'width': $container.outerWidth() + 'px',
                                'position': 'absolute',
                                'z-index': '1000000004'
                            });
                        }

                        console.log('✅ Z-index and positioning applied to Select2 dropdown elements');
                    }, 10);
                }
            });
        });

        // Aggressive overlay hiding function (defined globally)
        window.forceHideLoadingOverlay = function() {
            const overlay = document.getElementById('pageLoadingOverlay');
            if (overlay) {
                overlay.className = '';
                overlay.style.cssText = 'display: none !important; visibility: hidden !important; opacity: 0 !important; pointer-events: none !important;';
                console.log('🚀 FORCE HIDDEN: Loading overlay aggressively hidden');
            }
        };

        // Ensure loading overlay is hidden on window load
        $(window).on('load', function() {
            window.forceHideLoadingOverlay();
            console.log('✅ Page fully loaded, overlay force hidden');
        });

        // Also hide on page show (back button navigation)
        $(window).on('pageshow', function() {
            window.forceHideLoadingOverlay();
            console.log('✅ Page shown, overlay force hidden');
        });

        // Additional safeguard - check every 2 seconds and force hide if visible
        setInterval(function() {
            const overlay = document.getElementById('pageLoadingOverlay');
            if (overlay && window.getComputedStyle(overlay).display !== 'none') {
                console.warn('⚠️ Loading overlay detected visible, force hiding');
                window.forceHideLoadingOverlay();
            }
        }, 2000);

        // Auto submit form when per_page changes
        $('#per_page').change(function() {
            $('#filterForm').submit();
        });

        function deleteCategory(id) {
            if (confirm('Are you sure you want to delete this category?')) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: `{{ url('warehouse/item-categories') }}/${id}`,
                    type: 'DELETE',
                    success: function(response) {
                        if (response.success) {
                            showAlert('Category deleted successfully', 'success');
                            // Reload the page to refresh the list
                            setTimeout(function() {
                                window.location.reload();
                            }, 1500);
                        } else {
                            showAlert(response.message, 'error');
                        }
                    },
                    error: function(xhr) {
                        showAlert('Error deleting category', 'error');
                    }
                });
            }
        }

        function showAddCategoryModal() {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Clear form and validation errors
            $('#addCategoryForm')[0].reset();
            clearValidationErrors();

            // Load item types for dropdown
            loadItemTypesDropdown();

            // CUSTOM MODAL SHOW - Bypass Bootstrap completely to avoid aria-hidden conflicts
            const modal = $('#addCategoryModal');

            // Create and add backdrop manually
            const backdrop = $('<div class="modal-backdrop fade show"></div>');
            $('body').append(backdrop);

            // Set modal properties manually without Bootstrap
            modal.removeClass('fade').addClass('show');
            modal.css({
                'display': 'block',
                'z-index': 99999999999,
                'position': 'fixed',
                'top': '0',
                'left': '0',
                'width': '100%',
                'height': '100%',
                'padding-top': '80px',
                'overflow-y': 'auto'
            });

            // Never set aria-hidden, only set proper accessibility attributes
            modal.attr({
                'aria-modal': 'true',
                'role': 'dialog',
                'tabindex': '-1'
            });
            modal.removeAttr('aria-hidden');

            // Add body class and hide overlapping elements
            $('body').addClass('modal-open').css('overflow', 'hidden');

            // Hide all potential overlapping elements
            $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '1');

            // Set backdrop z-index
            backdrop.css('z-index', 99999999998);

            // Force hide any overlay elements
            $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').hide();
            $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').hide();

            // Initialize Select2 dropdowns after modal is shown
            setTimeout(function() {
                console.log('🔧 Reinitializing Select2 dropdowns in Add Category modal...');

                // Reinitialize Select2 dropdowns with proper dropdownParent and z-index
                $('#type_of_goods, #status').each(function() {
                    const $select = $(this);
                    const selectId = $select.attr('id');
                    const currentValue = $select.val();
                    const placeholder = $select.attr('data-placeholder') || $select.find('option:first').text() || 'Please select...';

                    console.log(`🔄 Processing ${selectId}, current value:`, currentValue);

                    // Destroy existing Select2
                    if ($select.hasClass('select2-hidden-accessible')) {
                        console.log(`🗑️ Destroying existing Select2 for ${selectId}`);
                        $select.select2('destroy');
                    }

                    // Reinitialize with proper positioning (attach to body, not modal)
                    console.log(`🆕 Reinitializing Select2 for ${selectId} with body attachment`);
                    $select.select2({
                        placeholder: placeholder,
                        allowClear: true,
                        theme: 'default',
                        width: '100%',
                        dropdownCssClass: 'modal-select2-dropdown',
                        containerCssClass: 'modal-select2-container',
                        dropdownAutoWidth: false
                    });

                    // Restore value
                    if (currentValue) {
                        $select.val(currentValue).trigger('change');
                    }
                });

                console.log('✅ Select2 reinitialization complete');
            }, 200);

            // Add close functionality to close button and backdrop
            modal.find('.close, [data-dismiss="modal"]').off('click.customModal').on('click.customModal', function(e) {
                e.preventDefault();
                closeAddCategoryModal();
            });

            // Close on backdrop click
            backdrop.off('click.customModal').on('click.customModal', function(e) {
                if (e.target === this) {
                    closeAddCategoryModal();
                }
            });

            // Close on escape key
            $(document).off('keydown.customModal').on('keydown.customModal', function(e) {
                if (e.key === 'Escape' && modal.hasClass('show')) {
                    closeAddCategoryModal();
                }
            });
        }

        function closeAddCategoryModal() {
            const modal = $('#addCategoryModal');

            // Remove modal classes and hide
            modal.removeClass('show').addClass('fade');
            modal.css('display', 'none');
            modal.removeAttr('aria-modal').removeAttr('role').removeAttr('tabindex');

            // Remove backdrop
            $('.modal-backdrop').remove();

            // Reset body state
            $('body').removeClass('modal-open').css('overflow', '');

            // Restore hidden elements
            $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '');
            $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').show();
            $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').show();

            // Remove event handlers
            $(document).off('keydown.customModal');
            modal.find('.close, [data-dismiss="modal"]').off('click.customModal');

            // Reset form
            $('#addCategoryForm')[0].reset();
            clearValidationErrors();
        }

        function saveCategory() {
            const form = $('#addCategoryForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveCategory()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.item-categories.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        // Show success alert
                        showAlert('Category added successfully!', 'success');

                        // Close modal using our custom function (no aria-hidden issues)
                        closeAddCategoryModal();

                        // Reload the page to refresh the list
                        setTimeout(function() {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showAlert(response.message || 'Error adding category', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        showAlert('Error adding category. Please try again.', 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            let alertClass = 'alert-success';
            let iconClass = 'check_circle';

            if (type === 'error' || type === 'danger') {
                alertClass = 'alert-danger';
                iconClass = 'error';
            } else if (type === 'warning') {
                alertClass = 'alert-warning';
                iconClass = 'warning';
            }

            // Convert newlines to HTML breaks for multi-line messages
            const formattedMessage = message.replace(/\n/g, '<br>');

            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 99999; max-width: 400px; white-space: pre-line;">
                    <div class="d-flex align-items-start">
                        <i class="material-icons mr-2" style="font-size: 20px;">${iconClass}</i>
                        <div class="flex-grow-1" style="font-size: 14px; line-height: 1.4;">
                            ${formattedMessage}
                        </div>
                    </div>
                    <button type="button" class="close ml-2" data-dismiss="alert" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
            `;

            $('body').append(alert);

            // Auto-close after longer time for detailed messages
            const timeout = message.includes('\n') ? 10000 : 5000;
            setTimeout(() => $('.alert').alert('close'), timeout);
        }

        function loadItemTypesDropdown() {
            $.ajax({
                url: '{{ route("warehouse.item-types.dropdown") }}',
                type: 'GET',
                success: function(response) {
                    if (response.success && response.data) {
                        const dropdown = $('#type_of_goods');

                        // Destroy Select2 if it exists
                        if (dropdown.hasClass('select2-hidden-accessible')) {
                            dropdown.select2('destroy');
                        }

                        dropdown.empty();
                        dropdown.append('<option value="">Select Type</option>');

                        response.data.forEach(function(itemType) {
                            dropdown.append(`<option value="${itemType.type_name}">${itemType.type_name}</option>`);
                        });

                        // Reinitialize Select2 with proper positioning
                        if ($('#addCategoryModal').hasClass('show')) {
                            dropdown.select2({
                                placeholder: 'Select Type',
                                allowClear: true,
                                theme: 'default',
                                width: '100%',
                                dropdownCssClass: 'modal-select2-dropdown',
                                containerCssClass: 'modal-select2-container',
                                dropdownAutoWidth: false
                            });
                        }
                    }
                },
                error: function(xhr) {
                    console.error('Failed to load item types:', xhr);
                    showAlert('Failed to load item types for dropdown', 'error');
                }
            });
        }

        // Import functionality
        function showImportModal() {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Clear form and results
            $('#importForm')[0].reset();
            $('#importResults').hide();
            $('#uploadProgress').hide();
            $('.custom-file-label').text('Choose file...');

            // CUSTOM MODAL SHOW - Bypass Bootstrap completely to avoid aria-hidden conflicts
            const modal = $('#importModal');

            // Create and add backdrop manually
            const backdrop = $('<div class="modal-backdrop fade show"></div>');
            $('body').append(backdrop);

            // Set modal properties manually without Bootstrap
            modal.removeClass('fade').addClass('show');
            modal.css({
                'display': 'block',
                'z-index': 99999999999,
                'position': 'fixed',
                'top': '0',
                'left': '0',
                'width': '100%',
                'height': '100%',
                'padding-top': '80px',
                'overflow-y': 'auto'
            });

            // Never set aria-hidden, only set proper accessibility attributes
            modal.attr({
                'aria-modal': 'true',
                'role': 'dialog',
                'tabindex': '-1'
            });
            modal.removeAttr('aria-hidden');

            // Add body class and hide overlapping elements
            $('body').addClass('modal-open').css('overflow', 'hidden');

            // Hide all potential overlapping elements
            $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '1');

            // Set backdrop z-index
            backdrop.css('z-index', 99999999998);

            // Force hide any overlay elements
            $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').hide();
            $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').hide();

            // Add close functionality to close button and backdrop
            modal.find('.close, [data-dismiss="modal"]').off('click.customModal').on('click.customModal', function(e) {
                e.preventDefault();
                closeImportModal();
            });

            // Close on backdrop click
            backdrop.off('click.customModal').on('click.customModal', function(e) {
                if (e.target === this) {
                    closeImportModal();
                }
            });
        }

        function closeImportModal() {
            const modal = $('#importModal');

            // Remove modal classes and hide
            modal.removeClass('show').addClass('fade');
            modal.css('display', 'none');
            modal.removeAttr('aria-modal').removeAttr('role').removeAttr('tabindex');

            // Remove backdrop
            $('.modal-backdrop').remove();

            // Reset body state
            $('body').removeClass('modal-open').css('overflow', '');

            // Restore hidden elements
            $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '');
            $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').show();
            $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').show();

            // Remove event handlers
            modal.find('.close, [data-dismiss="modal"]').off('click.customModal');

            // Reset form
            $('#importForm')[0].reset();
            $('#importResults').hide();
            $('#uploadProgress').hide();
            $('.custom-file-label').text('Choose file...');
        }

        function downloadTemplate() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Downloading...';
            btn.disabled = true;

            // Create download link - use route helper with proper authentication context
            const downloadUrl = '{{ route("warehouse.item-categories.template") }}';

            console.log('🔗 Attempting to download template from:', downloadUrl);

            fetch(downloadUrl, {
                method: 'GET',
                credentials: 'include',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
                .then(response => {
                    console.log('📡 Response received:', response.status, response.statusText);
                    if (!response.ok) {
                        throw new Error('Template download failed');
                    }
                    return response.blob();
                })
                .then(blob => {
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'item_categories_import_template.xlsx';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);

                    showAlert('Template downloaded successfully!', 'success');
                })
                .catch(error => {
                    console.error('💥 Download error:', error);
                    console.error('🔗 Failed URL:', downloadUrl);
                    showAlert('Failed to download template: ' + error.message, 'error');
                })
                .finally(() => {
                    btn.innerHTML = originalHtml;
                    btn.disabled = false;
                });
        }

        function importCategories() {
            const fileInput = $('#importFile')[0];

            if (!fileInput.files.length) {
                showAlert('Please select a file to import', 'error');
                return;
            }

            const formData = new FormData();
            formData.append('file', fileInput.files[0]);
            formData.append('_token', '{{ csrf_token() }}');

            // Show progress bar
            $('#uploadProgress').show();
            $('#importResults').hide();

            // Update import button
            const importBtn = $('#importBtn');
            const originalHtml = importBtn.html();
            importBtn.prop('disabled', true);
            importBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Importing...');

            $.ajax({
                url: '{{ route("warehouse.item-categories.import") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                xhr: function() {
                    var xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener("progress", function(evt) {
                        if (evt.lengthComputable) {
                            var percentComplete = (evt.loaded / evt.total) * 100;
                            $('.progress-bar').css('width', percentComplete + '%');
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    $('.progress-bar').css('width', '100%');

                    if (response.success) {
                        // First refresh table and show alert
                        categoriesTable.draw();

                        let message = `Import completed! `;
                        if (response.imported) message += `${response.imported} categories imported. `;
                        if (response.updated) message += `${response.updated} categories updated. `;

                        if (response.skipped && response.skipped > 0) {
                            message += `${response.skipped} rows skipped.`;

                            // Add detailed skip reasons if available
                            if (response.data && response.data.errors && response.data.errors.length > 0) {
                                const errors = response.data.errors;
                                message += `\n\nSkip reasons:\n`;
                                errors.slice(0, 5).forEach((error, index) => {
                                    message += `• ${error}\n`;
                                });

                                if (errors.length > 5) {
                                    message += `... and ${errors.length - 5} more errors.`;
                                }
                            }
                        }

                        showAlert(message, response.skipped > 0 ? 'warning' : 'success');

                        // Close modal using our custom function (no aria-hidden issues)
                        closeImportModal();

                        // Reload the page to refresh the list
                        setTimeout(function() {
                            window.location.reload();
                        }, response.skipped > 0 ? 3000 : 1500);
                    } else {
                        showAlert(response.message || 'Import failed', 'error');
                    }
                },
                error: function(xhr) {
                    let errorMessage = 'Import failed. Please check your file and try again.';

                    if (xhr.status === 422) {
                        const response = xhr.responseJSON;
                        if (response.errors) {
                            errorMessage = Object.values(response.errors).flat().join(' ');
                        } else if (response.message) {
                            errorMessage = response.message;
                        }
                    } else if (xhr.status === 413) {
                        errorMessage = 'File is too large. Please reduce file size and try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }

                    showAlert(errorMessage, 'error');
                },
                complete: function() {
                    // Reset import button
                    importBtn.prop('disabled', false);
                    importBtn.html(originalHtml);

                    // Hide progress bar after delay
                    setTimeout(() => {
                        $('#uploadProgress').hide();
                        $('.progress-bar').css('width', '0%');
                    }, 1000);
                }
            });
        }

        // File input label update
        $(document).on('change', '#importFile', function() {
            const fileName = $(this)[0].files[0]?.name || 'Choose file...';
            $(this).next('.custom-file-label').text(fileName);
        });
    </script>
@endpush