@extends('layouts.admin-simple')

@section('title', 'FOC Conversion History')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.inventory.index') }}">Inventory</a></li>
    <li class="breadcrumb-item active">FOC Conversion</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">FOC Conversion History</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Track conversions from good to damaged condition</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <a href="{{ route('warehouse.inventory.index') }}" class="btn btn-sm btn-outline-secondary mr-2">
                <i class="material-icons mr-1" style="font-size: 16px;">arrow_back</i>
                Back to Inventory
            </a>
            <button type="button" onclick="refreshTable()" class="btn btn-sm btn-outline-primary mr-2">
                <i class="material-icons mr-1" style="font-size: 16px;">refresh</i>
                Refresh
            </button>
            <button type="button" onclick="openNewConversionModal()" class="btn btn-sm btn-warning">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                New Conversion
            </button>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Conversions"
                :value="0"
                icon="transform"
                color="primary"
                subtitle="All time conversions"
                :loading="false"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="This Month"
                :value="0"
                icon="calendar_today"
                color="info"
                subtitle="Conversions this month"
                :loading="false"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Quantity"
                :value="0"
                icon="inventory"
                color="warning"
                subtitle="Total quantity converted"
                :loading="false"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Value Lost"
                :value="0"
                icon="attach_money"
                color="danger"
                subtitle="Estimated cost impact"
                :loading="false"
                :currency="true"
            />
        </div>
    </div>

    <!-- Export Buttons -->
    <div class="row warehouse-export-buttons mb-3">
        <div class="col-12 text-right">
            <button class="btn btn-success warehouse-export-btn" onclick="exportData('excel')">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">download</i>
                Export Excel
            </button>
            <button class="btn btn-info warehouse-export-btn" onclick="exportData('pdf')">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">picture_as_pdf</i>
                Export PDF
            </button>
            <button class="btn btn-secondary warehouse-export-btn" onclick="window.print()">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">print</i>
                Print
            </button>
        </div>
    </div>

    <!-- Conversion History Table -->
    <div class="card">
        <div class="card-header warehouse-report-header">
            <h5 class="mb-0 warehouse-report-title">
                <i class="material-icons align-middle mr-2">history</i>
                Conversion History
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="warehouse-table-container">
                <!-- Loading Overlay -->
                <div class="warehouse-loading-overlay" id="tableLoadingOverlay" style="display: none;">
                    <div class="warehouse-loading-content">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="sr-only">Loading...</span>
                            </div>
                            <div class="mt-2">Loading conversion history...</div>
                        </div>
                    </div>
                </div>

                <!-- Table -->
                <div class="warehouse-table-wrapper">
                    <table id="conversion-history-table" class="table table-hover mb-0 warehouse-table">
                        <thead class="thead-light">
                            <tr>
                                <th style="min-width: 100px;">Date</th>
                                <th style="min-width: 250px;">Item</th>
                                <th style="min-width: 100px;">Category</th>
                                <th style="min-width: 80px;" class="text-right">Quantity</th>
                                <th style="min-width: 100px;">Source</th>
                                <th style="min-width: 100px;">Status</th>
                                <th style="min-width: 100px;">User</th>
                                <th style="min-width: 100px;" class="text-right">Value Impact</th>
                                <th style="min-width: 150px;">Notes</th>
                                <th style="min-width: 80px;" class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be populated by DataTables -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Pagination will be added by DataTables -->
        <div class="card-footer" id="pagination-footer" style="display: none;">
            <div class="d-flex justify-content-between align-items-center">
                <div id="pagination-info">
                    <!-- Pagination info will be populated by DataTables -->
                </div>
                <div class="d-flex align-items-center">
                    <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                    <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                        <option value="10">10</option>
                        <option value="25">25</option>
                        <option value="50">50</option>
                        <option value="100">100</option>
                    </select>
                    <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
                </div>
            </div>
        </div>
    </div>

    <!-- Conversion Modal -->
    <div class="modal fade" id="conversionModal" tabindex="-1" role="dialog" aria-labelledby="conversionModalLabel">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header bg-warning text-white">
                    <h5 class="modal-title" id="conversionModalLabel">
                        <i class="material-icons mr-2">error</i>
                        Convert to Damaged Condition
                    </h5>
                    <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <form id="conversionForm" method="POST" action="{{ route('warehouse.inventory.condition-conversion.process') }}">
                    @csrf
                    <div class="modal-body">
                        <input type="hidden" name="inventory_id" id="inventory_id">

                        <div class="alert alert-warning">
                            <i class="material-icons align-middle mr-1">warning</i>
                            <strong>Warning:</strong> This action will permanently move the specified quantity from good inventory to damaged items. This cannot be undone.
                        </div>

                        <!-- Item Information Display -->
                        <div class="card bg-light mb-3">
                            <div class="card-body">
                                <h6 class="card-title">Item Information</h6>
                                <div id="itemInfo">
                                    <strong>Item:</strong> <span id="itemDescription">-</span><br>
                                    <strong>Available Quantity:</strong> <span id="availableQuantity">-</span><br>
                                    <strong>Current Location:</strong> <span id="currentLocation">-</span>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="quantity_to_convert">Quantity to Convert <span class="text-danger">*</span></label>
                                    <input type="number"
                                           name="quantity_to_convert"
                                           id="quantity_to_convert"
                                           class="form-control"
                                           step="0.01"
                                           min="0.01"
                                           required
                                           placeholder="Enter quantity">
                                    <small class="form-text text-muted">Maximum: <span id="maxQuantity">0</span></small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="damage_severity">Damage Severity <span class="text-danger">*</span></label>
                                    <select name="damage_severity" id="damage_severity" class="form-control" required>
                                        <option value="">Select Severity</option>
                                        <option value="minor">Minor Damage</option>
                                        <option value="moderate">Moderate Damage</option>
                                        <option value="severe">Severe Damage</option>
                                        <option value="total_loss">Total Loss</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="damage_description">Damage Description <span class="text-danger">*</span></label>
                            <textarea name="damage_description"
                                      id="damage_description"
                                      class="form-control"
                                      rows="3"
                                      required
                                      placeholder="Describe the nature of the damage or reason for conversion..."></textarea>
                        </div>

                        <div class="form-group">
                            <label for="cost_responsibility">Cost Responsibility <span class="text-danger">*</span></label>
                            <select name="cost_responsibility" id="cost_responsibility" class="form-control" required>
                                <option value="">Select Responsibility</option>
                                <option value="company">Company</option>
                                <option value="supplier">Supplier</option>
                                <option value="contractor">Contractor</option>
                                <option value="external">External Party</option>
                                <option value="unknown">Unknown/Investigation Required</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="damage_notes">Additional Notes</label>
                            <textarea name="damage_notes"
                                      id="damage_notes"
                                      class="form-control"
                                      rows="2"
                                      placeholder="Any additional notes or comments..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">
                            <i class="material-icons mr-1">cancel</i>
                            Cancel
                        </button>
                        <button type="submit" class="btn btn-warning">
                            <i class="material-icons mr-1">error</i>
                            Convert to Damaged
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
@endsection

<!-- Conversion Modal from Table -->
<div class="modal fade" id="conversionModal" tabindex="-1" role="dialog" aria-labelledby="conversionModalLabel">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-warning text-white">
                <h5 class="modal-title" id="conversionModalLabel">
                    <i class="material-icons mr-2">error</i>
                    Convert to Damaged Condition
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="conversionForm" method="POST" action="{{ route('warehouse.inventory.condition-conversion.process') }}">
                @csrf
                <div class="modal-body">
                    <input type="hidden" name="inventory_id" id="inventory_id">

                    <div class="alert alert-warning">
                        <i class="material-icons align-middle mr-1">warning</i>
                        <strong>Warning:</strong> This action will permanently move the specified quantity from good inventory to damaged items. This cannot be undone.
                    </div>

                    <!-- Item Information Display -->
                    <div class="card bg-light mb-3">
                        <div class="card-body">
                            <h6 class="card-title">Item Information</h6>
                            <div id="itemInfo">
                                <strong>Item:</strong> <span id="itemDescription">-</span><br>
                                <strong>Available Quantity:</strong> <span id="availableQuantity">-</span><br>
                                <strong>Current Location:</strong> <span id="currentLocation">-</span>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="quantity_to_convert">Quantity to Convert <span class="text-danger">*</span></label>
                        <input type="number"
                               name="quantity_to_convert"
                               id="quantity_to_convert"
                               class="form-control"
                               step="0.01"
                               min="0.01"
                               required
                               placeholder="Enter quantity">
                        <small class="form-text text-muted">Maximum: <span id="maxQuantity">0</span></small>
                    </div>

                    <!-- Hidden fields with default values -->
                    <input type="hidden" name="damage_severity" value="moderate">
                    <input type="hidden" name="damage_description" value="Warehouse FOC conversion">
                    <input type="hidden" name="cost_responsibility" value="company">

                    <div class="form-group">
                        <label for="damage_notes">Additional Notes</label>
                        <textarea name="damage_notes"
                                  id="damage_notes"
                                  class="form-control"
                                  rows="2"
                                  placeholder="Any additional notes or comments..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="material-icons mr-1">cancel</i>
                        Cancel
                    </button>
                    <button type="submit" class="btn btn-warning">
                        <i class="material-icons mr-1">error</i>
                        Convert to Damaged
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- New Conversion Modal -->
<div class="modal fade" id="newConversionModal" tabindex="-1" role="dialog" aria-labelledby="newConversionModalLabel">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-warning text-white">
                <h5 class="modal-title" id="newConversionModalLabel">
                    <i class="material-icons mr-2">add</i>
                    New FOC Conversion
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="newConversionForm" method="POST" action="{{ route('warehouse.inventory.condition-conversion.process') }}">
                @csrf
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="material-icons align-middle mr-1">info</i>
                        <strong>Quick Conversion:</strong> Search for an item and specify the quantity to convert from good to damaged condition.
                    </div>

                    <div class="form-group">
                        <label for="new_item_id">Select Item <span class="text-danger">*</span></label>
                        <select name="inventory_id" id="new_item_id" class="form-control" required>
                            <option value="">Search and select an item...</option>
                        </select>
                        <small class="form-text text-muted">Start typing to search for items with available inventory</small>
                    </div>

                    <!-- Item Information Display -->
                    <div id="newItemInfo" class="card bg-light mb-3" style="display: none;">
                        <div class="card-body">
                            <h6 class="card-title">Item Information</h6>
                            <div>
                                <strong>Item:</strong> <span id="newItemDescription">-</span><br>
                                <strong>Available Quantity:</strong> <span id="newAvailableQuantity">-</span><br>
                                <strong>Current Location:</strong> <span id="newCurrentLocation">-</span>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="new_quantity_to_convert">Quantity to Convert <span class="text-danger">*</span></label>
                        <input type="number"
                               name="quantity_to_convert"
                               id="new_quantity_to_convert"
                               class="form-control"
                               step="0.01"
                               min="0.01"
                               required
                               placeholder="Enter quantity">
                        <small class="form-text text-muted">Maximum: <span id="newMaxQuantity">0</span></small>
                    </div>

                    <!-- Hidden fields with default values -->
                    <input type="hidden" name="damage_severity" value="moderate">
                    <input type="hidden" name="damage_description" value="Warehouse FOC conversion">
                    <input type="hidden" name="cost_responsibility" value="company">

                    <div class="form-group">
                        <label for="new_damage_notes">Additional Notes</label>
                        <textarea name="damage_notes"
                                  id="new_damage_notes"
                                  class="form-control"
                                  rows="2"
                                  placeholder="Any additional notes or comments..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="material-icons mr-1">cancel</i>
                        Cancel
                    </button>
                    <button type="submit" class="btn btn-warning">
                        <i class="material-icons mr-1">error</i>
                        Convert to Damaged
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Status Modal -->
<div class="modal fade" id="editStatusModal" tabindex="-1" role="dialog" aria-labelledby="editStatusModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title" id="editStatusModalLabel">
                    <i class="material-icons mr-2">edit</i>
                    Update Damaged Item Status
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="editStatusForm">
                @csrf
                @method('PUT')
                <input type="hidden" id="damaged_item_id" name="damaged_item_id">

                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="material-icons align-middle mr-1">info</i>
                        <strong>Site Return Status Update:</strong> Update the status to reflect whether the project has covered the damaged item expense.
                    </div>

                    <!-- Item Information Display -->
                    <div class="card bg-light mb-3">
                        <div class="card-body">
                            <h6 class="card-title">Item Information</h6>
                            <div>
                                <strong>Item:</strong> <span id="statusItemDescription">-</span><br>
                                <strong>Source:</strong> <span class="badge badge-secondary">Site Return</span>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="status">Status <span class="text-danger">*</span></label>
                        <select name="status" id="status" class="form-control" required>
                            <option value="reported">Reported - Initial report of damage</option>
                            <option value="investigating">Investigating - Under review</option>
                            <option value="approved">Approved - Damage confirmed</option>
                            <option value="settled">Settled - Expense covered by project</option>
                            <option value="written_off">Written Off - Total loss</option>
                        </select>
                        <small class="form-text text-muted">Select "Settled" when the project has covered the expense for this damaged item.</small>
                    </div>

                    <div class="form-group">
                        <label for="notes">Additional Notes</label>
                        <textarea name="notes" id="status_notes" class="form-control" rows="3" placeholder="Add notes about the status update..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="material-icons mr-1">cancel</i>
                        Cancel
                    </button>
                    <button type="submit" class="btn btn-info">
                        <i class="material-icons mr-1">save</i>
                        Update Status
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Details Modal -->
<div class="modal fade" id="viewDetailsModal" tabindex="-1" role="dialog" aria-labelledby="viewDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="viewDetailsModalLabel">
                    <i class="material-icons mr-2">info</i>
                    Damaged Item Details
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="detailsLoadingSpinner" class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="sr-only">Loading...</span>
                    </div>
                    <div class="mt-2">Loading details...</div>
                </div>

                <div id="detailsContent" style="display: none;">
                    <div class="row">
                        <!-- Left Column -->
                        <div class="col-md-6">
                            <!-- Item Information -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-info" style="font-size: 18px;">inventory_2</i>
                                        Item Information
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <table class="table table-borderless table-sm mb-0">
                                        <tr>
                                            <td class="font-weight-medium">Description:</td>
                                            <td id="detail-item-description">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Item Code:</td>
                                            <td id="detail-item-code">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Category:</td>
                                            <td id="detail-item-category">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Unit of Measure:</td>
                                            <td id="detail-item-unit">-</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- Quantity & Cost Information -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-warning" style="font-size: 18px;">calculate</i>
                                        Quantity & Cost
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <table class="table table-borderless table-sm mb-0">
                                        <tr>
                                            <td class="font-weight-medium">Damaged Quantity:</td>
                                            <td><span id="detail-quantity" class="badge badge-danger">-</span></td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Unit Cost:</td>
                                            <td>AED <span id="detail-unit-cost">-</span></td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Total Cost:</td>
                                            <td><strong>AED <span id="detail-total-cost">-</span></strong></td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Recoverable Amount:</td>
                                            <td>AED <span id="detail-recoverable-amount">-</span></td>
                                        </tr>
                                        <tr class="table-warning">
                                            <td class="font-weight-bold">Total Loss:</td>
                                            <td><strong class="text-danger">AED <span id="detail-total-loss">-</span></strong></td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- Source Information -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-secondary" style="font-size: 18px;">source</i>
                                        Source Information
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <table class="table table-borderless table-sm mb-0">
                                        <tr>
                                            <td class="font-weight-medium">Source Type:</td>
                                            <td id="detail-source-type">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Source Reference:</td>
                                            <td id="detail-source-reference">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Inventory Reference:</td>
                                            <td id="detail-inventory-reference">-</td>
                                        </tr>
                                        <tr id="project-row" style="display: none;">
                                            <td class="font-weight-medium">Project:</td>
                                            <td id="detail-project">-</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Right Column -->
                        <div class="col-md-6">
                            <!-- Status & Responsibility -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-success" style="font-size: 18px;">assignment_turned_in</i>
                                        Status & Responsibility
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <table class="table table-borderless table-sm mb-0">
                                        <tr>
                                            <td class="font-weight-medium">Status:</td>
                                            <td id="detail-status">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Cost Responsibility:</td>
                                            <td id="detail-cost-responsibility">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Is Recoverable:</td>
                                            <td id="detail-is-recoverable">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Can Be Approved:</td>
                                            <td id="detail-can-be-approved">-</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- Damage Details -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-danger" style="font-size: 18px;">warning</i>
                                        Damage Details
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="form-group">
                                        <label class="font-weight-medium">Damage Severity:</label>
                                        <div id="detail-damage-severity">-</div>
                                    </div>
                                    <div class="form-group">
                                        <label class="font-weight-medium">Description:</label>
                                        <div id="detail-damage-description">-</div>
                                    </div>
                                    <div class="form-group mb-0">
                                        <label class="font-weight-medium">Notes:</label>
                                        <div id="detail-damage-notes">-</div>
                                    </div>
                                </div>
                            </div>

                            <!-- People & Dates -->
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-info" style="font-size: 18px;">people</i>
                                        People & Dates
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <table class="table table-borderless table-sm mb-0">
                                        <tr>
                                            <td class="font-weight-medium">Reported By:</td>
                                            <td id="detail-reported-by">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Reported At:</td>
                                            <td id="detail-reported-at">-</td>
                                        </tr>
                                        <tr id="approved-row" style="display: none;">
                                            <td class="font-weight-medium">Approved By:</td>
                                            <td id="detail-approved-by">-</td>
                                        </tr>
                                        <tr id="approved-at-row" style="display: none;">
                                            <td class="font-weight-medium">Approved At:</td>
                                            <td id="detail-approved-at">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Created At:</td>
                                            <td id="detail-created-at">-</td>
                                        </tr>
                                        <tr>
                                            <td class="font-weight-medium">Updated At:</td>
                                            <td id="detail-updated-at">-</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Resolution Notes (Full Width) -->
                    <div class="row" id="resolution-section" style="display: none;">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-success" style="font-size: 18px;">note</i>
                                        Resolution Notes
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div id="detail-resolution-notes" class="border rounded p-3 bg-light">-</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Disposal Information -->
                    <div class="row" id="disposal-section" style="display: none;">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <i class="material-icons mr-2 text-dark" style="font-size: 18px;">delete_forever</i>
                                        Disposal Information
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <strong>Disposal Method:</strong>
                                            <div id="detail-disposal-method">-</div>
                                        </div>
                                        <div class="col-md-6">
                                            <strong>Disposed At:</strong>
                                            <div id="detail-disposed-at">-</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="material-icons mr-1">close</i>
                    Close
                </button>
            </div>
        </div>
    </div>
</div>

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .warehouse-report-header {
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
        }

        .warehouse-report-title {
            font-size: 16px;
            font-weight: 600;
            color: #495057;
        }

        .warehouse-table-container {
            position: relative;
            overflow-x: auto;
        }

        .warehouse-table {
            white-space: nowrap;
        }

        .warehouse-table th {
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            border-top: none;
        }

        .warehouse-cell-nowrap {
            white-space: nowrap;
        }

        .warehouse-cell-date {
            font-weight: 500;
        }

        .warehouse-cell-number {
            font-weight: 600;
        }

        .warehouse-cell-currency {
            font-weight: 600;
        }

        .warehouse-badge {
            padding: 4px 8px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .warehouse-status-success {
            background-color: #d4edda;
            color: #155724;
        }

        .warehouse-status-warning {
            background-color: #fff3cd;
            color: #856404;
        }

        .warehouse-status-danger {
            background-color: #f8d7da;
            color: #721c24;
        }

        .warehouse-status-info {
            background-color: #d1ecf1;
            color: #0c5460;
        }

        .warehouse-status-primary {
            background-color: #cce5ff;
            color: #004085;
        }

        .warehouse-status-secondary {
            background-color: #e2e3e5;
            color: #383d41;
        }

        .warehouse-status-dark {
            background-color: #d6d8d9;
            color: #1b1e21;
        }

        .warehouse-empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }

        .warehouse-empty-state i {
            font-size: 48px;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        .warehouse-action-btn {
            padding: 2px 6px;
        }

        .warehouse-action-btn i {
            font-size: 16px;
        }

        .warehouse-export-buttons {
            margin-bottom: 20px;
        }

        .warehouse-export-btn {
            margin-left: 5px;
        }

        .warehouse-loading-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.9);
            z-index: 1000;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .modal-header.bg-warning {
            background-color: #ffc107 !important;
        }

        .alert-warning {
            color: #856404;
            background-color: #fff3cd;
            border-color: #ffeaa7;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            font-size: 0.75rem;
            font-weight: 500;
            line-height: 1;
            border-radius: 0.375rem;
        }

        .badge-info {
            color: #0c5460;
            background-color: #d1ecf1;
            border: 1px solid #bee5eb;
        }

        .badge-secondary {
            color: #383d41;
            background-color: #e2e3e5;
            border: 1px solid #d6d8db;
        }

        .badge-primary {
            color: #004085;
            background-color: #b3d4fc;
            border: 1px solid #9fcdff;
        }

        .text-danger {
            color: #dc3545 !important;
        }

        .text-success {
            color: #28a745 !important;
        }

        .text-muted {
            color: #6c757d !important;
        }

        /* Stats cards styling */
        .stats-card-value {
            transition: all 0.3s ease;
        }

        /* Animation for critical items */
        @keyframes pulseWarning {
            0% { background-color: transparent; }
            50% { background-color: rgba(255, 193, 7, 0.1); }
            100% { background-color: transparent; }
        }

        .pulse-warning {
            animation: pulseWarning 2s infinite;
        }


        /* Stats cards responsive layout - Better spacing for 4 cards */
        .row .col-lg-3 {
            display: flex;
            align-items: stretch;
        }

        /* Responsive breakpoints for better card spacing */
        @media (max-width: 1200px) {
            .row .col-lg-3 {
                flex: 0 0 50%;
                max-width: 50%;
            }
        }

        @media (max-width: 768px) {
            .row .col-lg-3 {
                flex: 0 0 100%;
                max-width: 100%;
            }
        }

        /* Stats card container spacing */
        .row.mb-4 {
            margin-left: -7.5px;
            margin-right: -7.5px;
        }

        .row.mb-4 > [class*="col-"] {
            padding-left: 7.5px;
            padding-right: 7.5px;
        }

        @media print {
            .warehouse-export-buttons,
            .warehouse-action-btn,
            .breadcrumb,
            .card-footer {
                display: none !important;
            }
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

    <script>
        let conversionTable;

        $(document).ready(function() {
            console.log('Document ready - initializing conversion history page');
            initializeDataTable();
            loadSummaryData();

            // Test if Select2 is loaded
            if (typeof $.fn.select2 === 'undefined') {
                console.error('Select2 is not loaded!');
            } else {
                console.log('Select2 is loaded successfully');
            }

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                conversionTable.page.len($(this).val()).draw();
            });
        });

        function initializeDataTable() {
            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            conversionTable = $('#conversion-history-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.inventory.condition-conversion") }}',
                    type: 'GET',
                    data: function(d) {
                        d.history_request = true;
                        return d;
                    },
                    headers: {
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    error: function(xhr, error, code) {
                        console.error('DataTable Ajax Error:', error, code);
                        console.error('Response Status:', xhr.status);
                        console.error('Response Text:', xhr.responseText);
                        showAlert('Error loading conversion history. Please refresh the page.', 'error');
                    }
                },
                columns: [
                    {
                        data: 'created_at',
                        name: 'created_at',
                        render: function(data) {
                            const date = new Date(data);
                            return `
                                <div class="warehouse-cell-date">
                                    ${date.toLocaleDateString()}
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'item_name',
                        render: function(data, type, row) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-warning text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 35px; height: 35px;">
                                            <i class="material-icons" style="font-size: 18px;">transform</i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${row.item_description}</div>
                                        <small class="text-muted">${row.item_code || 'No code'}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'category',
                        name: 'category',
                        render: function(data) {
                            return `
                                <span class="badge warehouse-badge warehouse-status-secondary">
                                    ${data || 'No Category'}
                                </span>
                            `;
                        }
                    },
                    {
                        data: 'quantity',
                        name: 'quantity',
                        className: 'text-right warehouse-cell-number',
                        render: function(data, type, row) {
                            return `
                                <strong>${parseFloat(data).toFixed(2)}</strong>
                                <br><small class="text-muted">${row.unit_of_measure || 'units'}</small>
                            `;
                        }
                    },
                    {
                        data: 'source',
                        name: 'source',
                        render: function(data) {
                            const sourceMap = {
                                'inventory_conversion': { text: 'Inventory Conversion', class: 'warehouse-status-info' },
                                'site_return': { text: 'Site Return', class: 'warehouse-status-secondary' },
                                'manual': { text: 'Manual Entry', class: 'warehouse-status-primary' }
                            };
                            const source = sourceMap[data] || { text: data, class: 'warehouse-status-secondary' };
                            return `<span class="badge warehouse-badge ${source.class}">${source.text}</span>`;
                        }
                    },
                    {
                        data: 'status',
                        name: 'status',
                        render: function(data, type, row) {
                            const statusMap = {
                                'reported': { text: 'Reported', class: 'warehouse-status-warning' },
                                'investigating': { text: 'Investigating', class: 'warehouse-status-info' },
                                'approved': { text: 'Approved', class: 'warehouse-status-primary' },
                                'settled': { text: 'Settled', class: 'warehouse-status-success' },
                                'written_off': { text: 'Written Off', class: 'warehouse-status-danger' }
                            };
                            const status = statusMap[data] || { text: data, class: 'warehouse-status-secondary' };
                            return `<span class="badge warehouse-badge ${status.class}">${status.text}</span>`;
                        }
                    },
                    {
                        data: 'user_name',
                        name: 'user_name',
                        render: function(data) {
                            return `<span>${data || 'System'}</span>`;
                        }
                    },
                    {
                        data: 'total_loss',
                        name: 'total_loss',
                        className: 'text-right warehouse-cell-currency',
                        render: function(data) {
                            return data ? `<strong>AED ${parseFloat(data).toFixed(2)}</strong>` : '<span class="text-muted">-</span>';
                        }
                    },
                    {
                        data: 'damage_description',
                        name: 'damage_description',
                        render: function(data, type, row) {
                            const notes = data || row.damage_notes || 'No notes provided';
                            return notes.length > 50 ?
                                `<span title="${notes}">${notes.substring(0, 50)}...</span>` :
                                notes;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        className: 'text-center',
                        render: function(data, type, row) {
                            let buttons = `
                                <button onclick="viewDetails(${data.id})" class="btn btn-sm btn-outline-primary warehouse-action-btn" title="View Details">
                                    <i class="material-icons">visibility</i>
                                </button>
                            `;

                            // Add print button
                            buttons += `
                                <button onclick="printConditionConversion(${data.id})" class="btn btn-sm btn-outline-secondary warehouse-action-btn ml-1" title="Print Conversion Report">
                                    <i class="material-icons">print</i>
                                </button>
                            `;

                            // Add edit status button only for site returns
                            if (row.source === 'site_return') {
                                buttons += `
                                    <button onclick="editStatus(${data.id}, '${row.status}', '${row.item_description}')" class="btn btn-sm btn-outline-warning warehouse-action-btn ml-1" title="Update Status">
                                        <i class="material-icons">edit</i>
                                    </button>
                                `;
                            }

                            return buttons;
                        }
                    }
                ],
                order: [[0, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: `
                        <div class="warehouse-empty-state">
                            <i class="material-icons">transform</i>
                            <h5>No Conversion History Found</h5>
                            <p class="text-muted">No FOC conversions have been recorded yet.</p>
                        </div>
                    `,
                    zeroRecords: `
                        <div class="warehouse-empty-state">
                            <i class="material-icons">search_off</i>
                            <h5>No Matching Conversions Found</h5>
                            <p class="text-muted">Try adjusting your search or filters.</p>
                        </div>
                    `
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                    updateSummaryCards();
                }
            });
        }

        function convertTodamaged(inventoryId, itemDescription) {
            // Reset form
            $('#conversionForm')[0].reset();
            $('#inventory_id').val(inventoryId);

            // Fetch and display item details
            $.get('{{ url("warehouse/inventory") }}/' + inventoryId + '/details', function(data) {
                $('#itemDescription').text(data.item_description);
                $('#availableQuantity').text(data.quantity_available + ' ' + data.unit);
                $('#currentLocation').text(data.location);
                $('#maxQuantity').text(data.quantity_available);
                $('#quantity_to_convert').attr('max', data.quantity_available);
            }).fail(function() {
                // Fallback if endpoint doesn't exist
                $('#itemDescription').text(itemDescription);
                $('#availableQuantity').text('Loading...');
                $('#currentLocation').text('Loading...');
            });

            $('#conversionModal').modal('show');
        }

        function refreshTable() {
            if (conversionTable) {
                conversionTable.ajax.reload();
                showAlert('Table refreshed successfully!', 'success');
            }
        }

        // Handle form submission
        $('#conversionForm').on('submit', function(e) {
            e.preventDefault();

            const formData = $(this).serialize();
            const submitBtn = $(this).find('button[type="submit"]');
            const originalBtnText = submitBtn.html();

            submitBtn.prop('disabled', true).html('<i class="material-icons mr-1">hourglass_empty</i> Processing...');

            $.post($(this).attr('action'), formData)
                .done(function(response) {
                    if (response.success) {
                        $('#conversionModal').modal('hide');
                        conversionTable.ajax.reload();
                        showAlert(response.message, 'success');
                    } else {
                        showAlert(response.message || 'An error occurred', 'error');
                    }
                })
                .fail(function(xhr) {
                    let message = 'An error occurred while processing the conversion';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        message = xhr.responseJSON.message;
                    }
                    showAlert(message, 'error');
                })
                .always(function() {
                    submitBtn.prop('disabled', false).html(originalBtnText);
                });
        });

        function openNewConversionModal() {
            console.log('Opening new conversion modal');
            // Reset form
            $('#newConversionForm')[0].reset();
            $('#newItemInfo').hide();

            // Test AJAX endpoint directly
            testSearchEndpoint();

            // Show modal first
            $('#newConversionModal').modal('show');
        }

        // Test function to verify the search endpoint
        function testSearchEndpoint() {
            $.ajax({
                url: '{{ route("warehouse.inventory.search-items") }}',
                type: 'GET',
                data: {
                    search: 'test',
                    with_inventory: true
                },
                success: function(data) {
                    console.log('Test search success:', data);
                },
                error: function(xhr, status, error) {
                    console.error('Test search failed:', xhr.responseText, status, error);
                }
            });
        }

        // Initialize Select2 when modal is shown
        $('#newConversionModal').on('shown.bs.modal', function() {
            // Destroy existing Select2 if it exists
            if ($('#new_item_id').hasClass('select2-hidden-accessible')) {
                $('#new_item_id').select2('destroy');
            }

            // Initialize Select2 for item selection
            $('#new_item_id').select2({
                dropdownParent: $('#newConversionModal'),
                placeholder: 'Search and select an item...',
                allowClear: true,
                width: '100%',
                ajax: {
                    url: '{{ route("warehouse.inventory.search-items") }}',
                    dataType: 'json',
                    delay: 300,
                    data: function(params) {
                        return {
                            search: params.term,
                            page: params.page || 1,
                            with_inventory: true
                        };
                    },
                    processResults: function(data, params) {
                        console.log('Search results:', data); // Debug log
                        if (!data.items) {
                            console.error('No items in response:', data);
                            return { results: [] };
                        }
                        return {
                            results: data.items.map(item => ({
                                id: item.inventory_id,
                                text: `${item.item_description} (${item.item_code}) - Available: ${item.quantity_available}`,
                                item_data: item
                            })),
                            pagination: {
                                more: data.has_more
                            }
                        };
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', xhr.responseText, status, error);
                    }
                },
                minimumInputLength: 0
            });

            // Handle item selection
            $('#new_item_id').on('select2:select', function(e) {
                const itemData = e.params.data.item_data;
                if (itemData) {
                    $('#newItemDescription').text(`${itemData.item_description} (${itemData.item_code})`);
                    $('#newAvailableQuantity').text(`${itemData.quantity_available} ${itemData.unit_of_measure || 'units'}`);
                    $('#newCurrentLocation').text(itemData.location || 'Main Warehouse');
                    $('#newMaxQuantity').text(itemData.quantity_available);
                    $('#new_quantity_to_convert').attr('max', itemData.quantity_available);
                    $('#newItemInfo').show();
                }
            });

            // Handle item deselection
            $('#new_item_id').on('select2:clear', function(e) {
                $('#newItemInfo').hide();
                $('#newMaxQuantity').text('0');
                $('#new_quantity_to_convert').attr('max', '');
            });
        });

        // Handle new conversion form submission
        $('#newConversionForm').on('submit', function(e) {
            e.preventDefault();

            const formData = $(this).serialize();
            const submitBtn = $(this).find('button[type="submit"]');
            const originalBtnText = submitBtn.html();

            submitBtn.prop('disabled', true).html('<i class="material-icons mr-1">hourglass_empty</i> Processing...');

            $.post($(this).attr('action'), formData)
                .done(function(response) {
                    if (response.success) {
                        $('#newConversionModal').modal('hide');
                        conversionTable.ajax.reload();
                        showAlert(response.message, 'success');

                        // Reset form
                        $('#newConversionForm')[0].reset();
                        $('#newItemInfo').hide();
                        $('#new_item_id').val(null).trigger('change');
                    } else {
                        showAlert(response.message || 'An error occurred', 'error');
                    }
                })
                .fail(function(xhr) {
                    let message = 'An error occurred while processing the conversion';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        message = xhr.responseJSON.message;
                    } else if (xhr.responseJSON && xhr.responseJSON.errors) {
                        const errors = Object.values(xhr.responseJSON.errors).flat();
                        message = errors.join('<br>');
                    }
                    showAlert(message, 'error');
                })
                .always(function() {
                    submitBtn.prop('disabled', false).html(originalBtnText);
                });
        });

        function loadSummaryData() {
            // Update summary cards when table data changes
            conversionTable.on('draw', function() {
                updateSummaryCards();
            });
        }

        function updateSummaryCards() {
            const tableData = conversionTable.rows().data();
            let totalConversions = tableData.length;
            let thisMonthCount = 0;
            let totalQuantity = 0;
            let totalValueLost = 0;

            const currentMonth = new Date().getMonth();
            const currentYear = new Date().getFullYear();

            for (let i = 0; i < tableData.length; i++) {
                const row = tableData[i];
                const createdDate = new Date(row.created_at);

                if (createdDate.getMonth() === currentMonth && createdDate.getFullYear() === currentYear) {
                    thisMonthCount++;
                }

                totalQuantity += parseFloat(row.quantity) || 0;
                totalValueLost += parseFloat(row.total_loss) || 0;
            }

            // Update stats cards with animation
            animateValue('.card:eq(0) .stats-card-value', totalConversions);
            animateValue('.card:eq(1) .stats-card-value', thisMonthCount);
            animateValue('.card:eq(2) .stats-card-value', totalQuantity);
            animateValueCurrency('.card:eq(3) .stats-card-value', totalValueLost);
        }

        function animateValue(selector, endValue) {
            const element = $(selector);
            const startValue = parseInt(element.text()) || 0;
            const duration = 1000;
            const stepTime = 50;
            const steps = duration / stepTime;
            const increment = (endValue - startValue) / steps;
            let currentValue = startValue;
            let step = 0;

            const timer = setInterval(() => {
                currentValue += increment;
                step++;

                if (step >= steps) {
                    currentValue = endValue;
                    clearInterval(timer);
                }

                element.text(Math.round(currentValue));
            }, stepTime);
        }

        function animateValueCurrency(selector, endValue) {
            const element = $(selector);
            // Extract numeric value from existing content
            const currentText = element.text().replace(/[^\d.]/g, '');
            const startValue = parseFloat(currentText) || 0;
            const duration = 1000;
            const stepTime = 50;
            const steps = duration / stepTime;
            const increment = (endValue - startValue) / steps;
            let currentValue = startValue;
            let step = 0;

            const timer = setInterval(() => {
                currentValue += increment;
                step++;

                if (step >= steps) {
                    currentValue = endValue;
                    clearInterval(timer);
                }

                element.text(currentValue.toFixed(2));
            }, stepTime);
        }

        function viewDetails(damagedItemId) {
            // Show modal and loading spinner
            $('#viewDetailsModal').modal('show');
            $('#detailsLoadingSpinner').show();
            $('#detailsContent').hide();

            // Fetch details
            $.ajax({
                url: `{{ url('warehouse/inventory/damaged-items') }}/${damagedItemId}`,
                type: 'GET',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            })
            .done(function(response) {
                if (response.success) {
                    populateDetailsModal(response.data);
                    $('#detailsLoadingSpinner').hide();
                    $('#detailsContent').show();
                } else {
                    showAlert(response.message || 'Failed to load details', 'error');
                    $('#viewDetailsModal').modal('hide');
                }
            })
            .fail(function(xhr) {
                let message = 'An error occurred while loading details';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                }
                showAlert(message, 'error');
                $('#viewDetailsModal').modal('hide');
            });
        }

        function populateDetailsModal(data) {
            // Item Information
            $('#detail-item-description').text(data.item.description);
            $('#detail-item-code').text(data.item.code);
            $('#detail-item-category').text(data.item.category);
            $('#detail-item-unit').text(data.item.unit);

            // Quantity & Cost
            $('#detail-quantity').text(`${data.quantity} ${data.unit}`);
            $('#detail-unit-cost').text(parseFloat(data.unit_cost || 0).toFixed(2));
            $('#detail-total-cost').text(parseFloat(data.total_cost || 0).toFixed(2));
            $('#detail-recoverable-amount').text(parseFloat(data.recoverable_amount || 0).toFixed(2));
            $('#detail-total-loss').text(parseFloat(data.total_loss || 0).toFixed(2));

            // Source Information
            const sourceMap = {
                'inventory_conversion': '<span class="badge badge-info">Inventory Conversion</span>',
                'site_return': '<span class="badge badge-secondary">Site Return</span>',
                'manual': '<span class="badge badge-primary">Manual Entry</span>'
            };
            $('#detail-source-type').html(sourceMap[data.source_type] || data.source_type);
            $('#detail-source-reference').text(data.source_reference || '-');
            $('#detail-inventory-reference').text(data.inventory_reference || '-');

            // Project (if exists)
            if (data.project) {
                $('#detail-project').html(`
                    <strong>${data.project.name}</strong><br>
                    <small class="text-muted">${data.project.number}</small>
                    ${data.project.details ? '<br><em>' + data.project.details + '</em>' : ''}
                `);
                $('#project-row').show();
            } else {
                $('#project-row').hide();
            }

            // Status & Responsibility
            const statusMap = {
                'reported': '<span class="badge badge-warning">Reported</span>',
                'investigating': '<span class="badge badge-info">Investigating</span>',
                'approved': '<span class="badge badge-primary">Approved</span>',
                'settled': '<span class="badge badge-success">Settled</span>',
                'written_off': '<span class="badge badge-danger">Written Off</span>'
            };
            $('#detail-status').html(statusMap[data.status] || data.status_text);

            // Cost Responsibility - show project name if responsibility is 'project'
            let costResponsibilityDisplay = data.cost_responsibility_text;
            if (data.cost_responsibility === 'project' && data.project && data.project.name) {
                costResponsibilityDisplay = `Project: ${data.project.name}`;
            }
            $('#detail-cost-responsibility').text(costResponsibilityDisplay);
            $('#detail-is-recoverable').html(data.is_recoverable ?
                '<span class="badge badge-success">Yes</span>' :
                '<span class="badge badge-secondary">No</span>'
            );
            $('#detail-can-be-approved').html(data.can_be_approved ?
                '<span class="badge badge-warning">Yes</span>' :
                '<span class="badge badge-secondary">No</span>'
            );

            // Damage Details
            const severityMap = {
                'minor': '<span class="badge badge-warning">Minor</span>',
                'moderate': '<span class="badge badge-warning">Moderate</span>',
                'severe': '<span class="badge badge-danger">Severe</span>',
                'total_loss': '<span class="badge badge-dark">Total Loss</span>'
            };
            $('#detail-damage-severity').html(severityMap[data.damage.severity] || data.damage.severity || '-');
            $('#detail-damage-description').text(data.damage.description || 'No description provided');
            $('#detail-damage-notes').text(data.damage.notes || 'No additional notes');

            // People & Dates
            $('#detail-reported-by').html(`
                <strong>${data.reported_by.name}</strong>
                ${data.reported_by.email ? '<br><small class="text-muted">' + data.reported_by.email + '</small>' : ''}
            `);
            $('#detail-reported-at').text(data.reported_by.at || '-');

            // Approved info (if exists)
            if (data.approved_by) {
                $('#detail-approved-by').html(`
                    <strong>${data.approved_by.name}</strong>
                    ${data.approved_by.email ? '<br><small class="text-muted">' + data.approved_by.email + '</small>' : ''}
                `);
                $('#detail-approved-at').text(data.approved_by.at || '-');
                $('#approved-row, #approved-at-row').show();
            } else {
                $('#approved-row, #approved-at-row').hide();
            }

            $('#detail-created-at').text(data.created_at);
            $('#detail-updated-at').text(data.updated_at);

            // Resolution Notes (if exists)
            if (data.resolution_notes) {
                $('#detail-resolution-notes').text(data.resolution_notes);
                $('#resolution-section').show();
            } else {
                $('#resolution-section').hide();
            }

            // Disposal Information (if exists)
            if (data.disposal_method || data.disposed_at) {
                $('#detail-disposal-method').text(data.disposal_method || '-');
                $('#detail-disposed-at').text(data.disposed_at || '-');
                $('#disposal-section').show();
            } else {
                $('#disposal-section').hide();
            }
        }

        function editStatus(damagedItemId, currentStatus, itemDescription) {
            // Set form values
            $('#editStatusForm #damaged_item_id').val(damagedItemId);
            $('#editStatusForm #status').val(currentStatus);
            $('#statusItemDescription').text(itemDescription);

            // Show modal
            $('#editStatusModal').modal('show');
        }

        // Handle edit status form submission
        $(document).on('submit', '#editStatusForm', function(e) {
            e.preventDefault();

            const damagedItemId = $('#damaged_item_id').val();
            const formData = $(this).serialize();
            const submitBtn = $(this).find('button[type="submit"]');
            const originalBtnText = submitBtn.html();

            submitBtn.prop('disabled', true).html('<i class="material-icons mr-1">hourglass_empty</i> Updating...');

            $.ajax({
                url: `{{ url('warehouse/inventory/damaged-items') }}/${damagedItemId}/status`,
                type: 'PUT',
                data: formData,
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            })
            .done(function(response) {
                if (response.success) {
                    $('#editStatusModal').modal('hide');
                    conversionTable.ajax.reload();
                    showAlert(response.message, 'success');

                    // Reset form
                    $('#editStatusForm')[0].reset();
                } else {
                    showAlert(response.message || 'An error occurred', 'error');
                }
            })
            .fail(function(xhr) {
                let message = 'An error occurred while updating status';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                }
                showAlert(message, 'error');
            })
            .always(function() {
                submitBtn.prop('disabled', false).html(originalBtnText);
            });
        });

        function exportData(format) {
            const currentUrl = new URL(window.location.href);
            currentUrl.searchParams.set('export', format);
            window.location.href = currentUrl.toString();
        }

        function printConditionConversion(conversionId) {
            // Open print window for individual conversion
            const printUrl = '{{ route("warehouse.inventory.condition-conversion.print", ":id") }}'.replace(':id', conversionId);
            window.open(printUrl, '_blank', 'width=1200,height=800');
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : type === 'info' ? 'alert-info' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }

        // Handle modal accessibility and cleanup
        $('.modal').on('hidden.bs.modal', function() {
            // Remove any lingering focus and reset aria-hidden
            $(this).attr('aria-hidden', 'true');
            $(this).find('button').blur();
            $('body').removeClass('modal-open');
        });

        $('.modal').on('shown.bs.modal', function() {
            $(this).attr('aria-hidden', 'false');
        });
    </script>
@endpush