@extends('layouts.admin-simple')

@section('title', 'Department Management')
@section('page-title', 'Department Management')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item active" aria-current="page">Departments</li>
@endsection

@section('page-actions')
    <div class="ml-auto">
        <button class="btn btn-primary" data-toggle="modal" data-target="#createDepartmentModal">
            <i class="material-icons">add</i> Add Department
        </button>
    </div>
@endsection

@section('content')
    <div class="card">
        <div class="card-header card-header-large bg-white">
            <h4 class="card-header__title">Departments</h4>
            <div class="card-subtitle text-muted">Manage organizational departments</div>
        </div>
        
        <div class="table-responsive">
            <table class="table mb-0">
                <thead>
                    <tr>
                        <th>Department</th>
                        <th>Code</th>
                        <th>Manager</th>
                        <th>Users</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($departments as $department)
                    <tr>
                        <td>
                            <div>
                                <strong>{{ $department->name }}</strong>
                                @if($department->description)
                                    <div class="text-muted small">{{ $department->description }}</div>
                                @endif
                            </div>
                        </td>
                        <td>{{ $department->code ?? '-' }}</td>
                        <td>
                            @if($department->manager_name)
                                <div>{{ $department->manager_name }}</div>
                                @if($department->manager_email)
                                    <div class="text-muted small">{{ $department->manager_email }}</div>
                                @endif
                            @else
                                <span class="text-muted">-</span>
                            @endif
                        </td>
                        <td>
                            <span class="badge badge-light">{{ $department->users_count }} users</span>
                        </td>
                        <td>
                            @if($department->is_active)
                                <span class="badge badge-soft-success">Active</span>
                            @else
                                <span class="badge badge-soft-danger">Inactive</span>
                            @endif
                        </td>
                        <td>
                            <div class="dropdown">
                                <a href="#" class="dropdown-toggle" data-toggle="dropdown" data-caret="false">
                                    <i class="material-icons">more_vert</i>
                                </a>
                                <div class="dropdown-menu dropdown-menu-right">
                                    <a href="#" class="dropdown-item" onclick="editDepartment({{ $department->id }})">
                                        <i class="material-icons mr-2">edit</i> Edit
                                    </a>
                                    @if($department->users_count == 0)
                                        <div class="dropdown-divider"></div>
                                        <a href="#" class="dropdown-item text-danger" 
                                           data-toggle="modal" data-target="#deleteDepartmentModal{{ $department->id }}">
                                            <i class="material-icons mr-2">delete</i> Delete
                                        </a>
                                    @else
                                        <div class="dropdown-divider"></div>
                                        <span class="dropdown-item text-muted">
                                            <i class="material-icons mr-2">lock</i> Cannot delete (has users)
                                        </span>
                                    @endif
                                </div>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="6" class="text-center py-4">
                            <div class="empty-state">
                                <i class="material-icons mb-3" style="font-size: 48px; color: #ddd;">business</i>
                                <h5 class="text-muted">No departments found</h5>
                                <p class="text-muted">Start by creating your first department.</p>
                                <button class="btn btn-primary" data-toggle="modal" data-target="#createDepartmentModal">
                                    <i class="material-icons">add</i> Create First Department
                                </button>
                            </div>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
@endsection

@section('modals')
<!-- Create Department Modal -->
<div class="modal fade" id="createDepartmentModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Department</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form method="POST" action="{{ route('admin.departments.store') }}">
                @csrf
                <div class="modal-body">
                    <div class="form-group">
                        <label for="name">Department Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="name" name="name" required>
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="code">Department Code</label>
                        <input type="text" class="form-control" id="code" name="code" placeholder="e.g., HR, IT, FIN">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="2"></textarea>
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="manager_name">Manager Name</label>
                        <input type="text" class="form-control" id="manager_name" name="manager_name">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="manager_email">Manager Email</label>
                        <input type="email" class="form-control" id="manager_email" name="manager_email">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <div class="custom-control custom-checkbox">
                            <input type="checkbox" class="custom-control-input" id="is_active" name="is_active" checked>
                            <label class="custom-control-label" for="is_active">Department is Active</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons mr-2">save</i> Create Department
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Department Modal -->
<div class="modal fade" id="editDepartmentModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Department</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form method="POST" action="#" id="editDepartmentForm">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="form-group">
                        <label for="edit_name">Department Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit_name" name="name" required>
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_code">Department Code</label>
                        <input type="text" class="form-control" id="edit_code" name="code">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_description">Description</label>
                        <textarea class="form-control" id="edit_description" name="description" rows="2"></textarea>
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_manager_name">Manager Name</label>
                        <input type="text" class="form-control" id="edit_manager_name" name="manager_name">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_manager_email">Manager Email</label>
                        <input type="email" class="form-control" id="edit_manager_email" name="manager_email">
                        <div class="invalid-feedback"></div>
                    </div>
                    
                    <div class="form-group">
                        <div class="custom-control custom-checkbox">
                            <input type="checkbox" class="custom-control-input" id="edit_is_active" name="is_active">
                            <label class="custom-control-label" for="edit_is_active">Department is Active</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons mr-2">save</i> Update Department
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modals -->
@foreach($departments as $department)
    @if($department->users_count == 0)
    <div class="modal fade" id="deleteDepartmentModal{{ $department->id }}" tabindex="-1" role="dialog" aria-labelledby="deleteDepartmentModalLabel{{ $department->id }}" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered" role="document">
            <div class="modal-content border-0 shadow">
                <div class="modal-header border-0 pb-0">
                    <h5 class="modal-title font-weight-bold" id="deleteDepartmentModalLabel{{ $department->id }}">Delete Department</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body pt-2">
                    <div class="text-center">
                        <div class="mb-3">
                            <div class="bg-danger-light d-flex align-items-center justify-content-center rounded-circle mx-auto" style="width: 64px; height: 64px; background-color: rgba(220, 53, 69, 0.1);">
                                <i class="material-icons text-danger" style="font-size: 32px;">warning</i>
                            </div>
                        </div>
                        <h5 class="mb-3">Are you sure?</h5>
                        <p class="text-muted mb-3">You are about to delete the department:</p>
                        <div class="card border-left-danger shadow-sm mb-3">
                            <div class="card-body py-3">
                                <div class="d-flex align-items-center">
                                    <div class="bg-danger-light d-flex align-items-center justify-content-center rounded mr-3" style="width: 48px; height: 48px; background-color: rgba(220, 53, 69, 0.1);">
                                        <i class="material-icons text-danger">business</i>
                                    </div>
                                    <div>
                                        <strong>{{ $department->name }}</strong>
                                        @if($department->description)
                                            <div class="text-muted small">{{ Str::limit($department->description, 50) }}</div>
                                        @endif
                                        @if($department->code)
                                            <div class="text-muted small">Code: {{ $department->code }}</div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                        <p class="text-muted small">
                            <i class="material-icons" style="font-size: 16px; vertical-align: middle;">info_outline</i>
                            This action cannot be undone. The department and all associated data will be permanently deleted.
                        </p>
                    </div>
                </div>
                <div class="modal-footer border-0 pt-0">
                    <button type="button" class="btn btn-outline-secondary" data-dismiss="modal">
                        <i class="material-icons mr-1" style="font-size: 18px;">close</i>
                        Cancel
                    </button>
                    <form method="POST" action="{{ route('admin.departments.delete', $department->id) }}" style="display: inline;">
                        @csrf
                        @method('DELETE')
                        <button type="submit" class="btn btn-danger">
                            <i class="material-icons mr-1" style="font-size: 18px;">delete</i>
                            Delete Department
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    @endif
@endforeach
@endsection

@push('scripts')
<script>
// Edit Department
function editDepartment(id) {
    // Fetch department data
    const department = @json($departments);
    const dept = department.find(d => d.id === id);
    
    if (dept) {
        const form = document.getElementById('editDepartmentForm');
        form.action = `/admin/departments/${dept.id}`;
        
        document.getElementById('edit_name').value = dept.name;
        document.getElementById('edit_code').value = dept.code || '';
        document.getElementById('edit_description').value = dept.description || '';
        document.getElementById('edit_manager_name').value = dept.manager_name || '';
        document.getElementById('edit_manager_email').value = dept.manager_email || '';
        document.getElementById('edit_is_active').checked = dept.is_active;
        
        $('#editDepartmentModal').modal('show');
    }
}

// Reset form on modal close
$('#createDepartmentModal').on('hidden.bs.modal', function() {
    document.getElementById('createDepartmentForm').reset();
});
</script>
@endpush