<?php

require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use App\Models\DirectDeliveryItem;

try {
    echo "=== Direct Delivery Return Quantities Fix Script ===" . PHP_EOL;
    echo "Finding completed site returns with direct delivery item references..." . PHP_EOL . PHP_EOL;

    // Find all completed site returns that have direct_delivery_item_id
    $siteReturnItems = IncomingOperationItem::whereHas('operation', function($query) {
            $query->where('operation_type', 'site_return')
                  ->where('status', 'completed');
        })
        ->whereNotNull('direct_delivery_item_id')
        ->where('quantity_accepted', '>', 0)
        ->get();

    echo "Found " . $siteReturnItems->count() . " site return items to process..." . PHP_EOL . PHP_EOL;

    if ($siteReturnItems->isEmpty()) {
        echo "No site return items found to fix." . PHP_EOL;
        exit;
    }

    $processedCount = 0;
    $fixedCount = 0;
    $errorCount = 0;

    foreach ($siteReturnItems as $returnItem) {
        echo "Processing Site Return Item: Operation {$returnItem->operation->operation_number}" . PHP_EOL;
        echo "  Item ID: {$returnItem->item_id}, Quantity Accepted: {$returnItem->quantity_accepted}" . PHP_EOL;
        echo "  Direct Delivery Item ID: {$returnItem->direct_delivery_item_id}" . PHP_EOL;

        try {
            $directDeliveryItem = DirectDeliveryItem::find($returnItem->direct_delivery_item_id);

            if (!$directDeliveryItem) {
                echo "  ERROR: Direct delivery item not found!" . PHP_EOL;
                $errorCount++;
                continue;
            }

            $oldReturnedQty = $directDeliveryItem->returned_quantity ?? 0;

            // Update the returned quantity
            $directDeliveryItem->recordReturn($returnItem->quantity_accepted);

            $newReturnedQty = $directDeliveryItem->returned_quantity;

            echo "  ✓ Updated: Returned quantity {$oldReturnedQty} + {$returnItem->quantity_accepted} = {$newReturnedQty}" . PHP_EOL;
            echo "  Available for return: {$directDeliveryItem->getAvailableForReturnQuantity()}" . PHP_EOL;

            $fixedCount++;

        } catch (Exception $e) {
            echo "  ERROR: " . $e->getMessage() . PHP_EOL;
            $errorCount++;
        }

        $processedCount++;
        echo PHP_EOL;
    }

    echo "=== Fix Complete ===" . PHP_EOL;
    echo "Items processed: {$processedCount}" . PHP_EOL;
    echo "Items fixed successfully: {$fixedCount}" . PHP_EOL;
    echo "Errors encountered: {$errorCount}" . PHP_EOL;

    // Show summary of direct deliveries with returns
    echo PHP_EOL . "=== Direct Delivery Return Summary ===" . PHP_EOL;
    $directDeliveries = DB::table('direct_deliveries as dd')
        ->join('direct_delivery_items as ddi', 'dd.id', '=', 'ddi.direct_delivery_id')
        ->where('ddi.returned_quantity', '>', 0)
        ->select(
            'dd.delivery_number',
            'dd.id as delivery_id',
            DB::raw('COUNT(ddi.id) as items_with_returns'),
            DB::raw('SUM(ddi.received_quantity) as total_received'),
            DB::raw('SUM(ddi.returned_quantity) as total_returned')
        )
        ->groupBy('dd.id', 'dd.delivery_number')
        ->get();

    foreach ($directDeliveries as $delivery) {
        echo "Delivery {$delivery->delivery_number}: {$delivery->items_with_returns} items, ";
        echo "Received: {$delivery->total_received}, Returned: {$delivery->total_returned}" . PHP_EOL;
    }

} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . PHP_EOL;
    echo "Trace: " . $e->getTraceAsString() . PHP_EOL;
}