<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class WarehouseRolesAndPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Warehouse-specific permissions
        $permissions = [
            // Dashboard permissions
            [
                'name' => 'warehouse.dashboard.view',
                'display_name' => 'View Warehouse Dashboard',
                'description' => 'Can view warehouse dashboard and analytics',
                'group' => 'warehouse_dashboard'
            ],

            // Supplier Management
            [
                'name' => 'warehouse.suppliers.view',
                'display_name' => 'View Suppliers',
                'description' => 'Can view supplier list and details',
                'group' => 'warehouse_suppliers'
            ],
            [
                'name' => 'warehouse.suppliers.create',
                'display_name' => 'Create Suppliers',
                'description' => 'Can create new suppliers',
                'group' => 'warehouse_suppliers'
            ],
            [
                'name' => 'warehouse.suppliers.edit',
                'display_name' => 'Edit Suppliers',
                'description' => 'Can edit supplier information',
                'group' => 'warehouse_suppliers'
            ],
            [
                'name' => 'warehouse.suppliers.delete',
                'display_name' => 'Delete Suppliers',
                'description' => 'Can delete suppliers',
                'group' => 'warehouse_suppliers'
            ],

            // Project Management
            [
                'name' => 'warehouse.projects.view',
                'display_name' => 'View Projects',
                'description' => 'Can view project list and details',
                'group' => 'warehouse_projects'
            ],
            [
                'name' => 'warehouse.projects.create',
                'display_name' => 'Create Projects',
                'description' => 'Can create new projects',
                'group' => 'warehouse_projects'
            ],
            [
                'name' => 'warehouse.projects.edit',
                'display_name' => 'Edit Projects',
                'description' => 'Can edit project information',
                'group' => 'warehouse_projects'
            ],
            [
                'name' => 'warehouse.projects.delete',
                'display_name' => 'Delete Projects',
                'description' => 'Can delete projects',
                'group' => 'warehouse_projects'
            ],

            // Items Management
            [
                'name' => 'warehouse.items.view',
                'display_name' => 'View Items',
                'description' => 'Can view item catalog and details',
                'group' => 'warehouse_items'
            ],
            [
                'name' => 'warehouse.items.create',
                'display_name' => 'Create Items',
                'description' => 'Can create new items',
                'group' => 'warehouse_items'
            ],
            [
                'name' => 'warehouse.items.edit',
                'display_name' => 'Edit Items',
                'description' => 'Can edit item information',
                'group' => 'warehouse_items'
            ],
            [
                'name' => 'warehouse.items.delete',
                'display_name' => 'Delete Items',
                'description' => 'Can delete items',
                'group' => 'warehouse_items'
            ],

            // Inventory Management
            [
                'name' => 'warehouse.inventory.view',
                'display_name' => 'View Inventory',
                'description' => 'Can view inventory levels and stock information',
                'group' => 'warehouse_inventory'
            ],
            [
                'name' => 'warehouse.inventory.adjust',
                'display_name' => 'Adjust Inventory',
                'description' => 'Can make inventory adjustments',
                'group' => 'warehouse_inventory'
            ],
            [
                'name' => 'warehouse.inventory.transfer',
                'display_name' => 'Transfer Inventory',
                'description' => 'Can transfer inventory between divisions',
                'group' => 'warehouse_inventory'
            ],

            // Incoming Transactions (Purchase/Receipt)
            [
                'name' => 'warehouse.incoming.view',
                'display_name' => 'View Incoming Transactions',
                'description' => 'Can view incoming goods transactions',
                'group' => 'warehouse_incoming'
            ],
            [
                'name' => 'warehouse.incoming.create',
                'display_name' => 'Create Incoming Transactions',
                'description' => 'Can create incoming goods receipts',
                'group' => 'warehouse_incoming'
            ],
            [
                'name' => 'warehouse.incoming.edit',
                'display_name' => 'Edit Incoming Transactions',
                'description' => 'Can edit incoming transactions',
                'group' => 'warehouse_incoming'
            ],
            [
                'name' => 'warehouse.incoming.complete',
                'display_name' => 'Complete Incoming Transactions',
                'description' => 'Can complete and finalize incoming transactions',
                'group' => 'warehouse_incoming'
            ],

            // Outgoing Transactions (Material Requests)
            [
                'name' => 'warehouse.outgoing.view',
                'display_name' => 'View Outgoing Transactions',
                'description' => 'Can view material requests and outgoing transactions',
                'group' => 'warehouse_outgoing'
            ],
            [
                'name' => 'warehouse.outgoing.create',
                'display_name' => 'Create Material Requests',
                'description' => 'Can create material requests',
                'group' => 'warehouse_outgoing'
            ],
            [
                'name' => 'warehouse.outgoing.approve',
                'display_name' => 'Approve Material Requests',
                'description' => 'Can approve material requests',
                'group' => 'warehouse_outgoing'
            ],
            [
                'name' => 'warehouse.outgoing.fulfill',
                'display_name' => 'Fulfill Material Requests',
                'description' => 'Can fulfill and release materials',
                'group' => 'warehouse_outgoing'
            ],

            // Return Transactions
            [
                'name' => 'warehouse.returns.view',
                'display_name' => 'View Return Transactions',
                'description' => 'Can view return transactions',
                'group' => 'warehouse_returns'
            ],
            [
                'name' => 'warehouse.returns.create',
                'display_name' => 'Create Return Transactions',
                'description' => 'Can create return transactions',
                'group' => 'warehouse_returns'
            ],
            [
                'name' => 'warehouse.returns.process',
                'display_name' => 'Process Returns',
                'description' => 'Can process returned goods',
                'group' => 'warehouse_returns'
            ],

            // Reports
            [
                'name' => 'warehouse.reports.view',
                'display_name' => 'View Warehouse Reports',
                'description' => 'Can view warehouse reports',
                'group' => 'warehouse_reports'
            ],
            [
                'name' => 'warehouse.reports.export',
                'display_name' => 'Export Reports',
                'description' => 'Can export reports to various formats',
                'group' => 'warehouse_reports'
            ],
            [
                'name' => 'warehouse.reports.advanced',
                'display_name' => 'Advanced Reports',
                'description' => 'Can access advanced reporting features',
                'group' => 'warehouse_reports'
            ],

            // Settings and Configuration
            [
                'name' => 'warehouse.settings.view',
                'display_name' => 'View Warehouse Settings',
                'description' => 'Can view warehouse configuration',
                'group' => 'warehouse_settings'
            ],
            [
                'name' => 'warehouse.settings.manage',
                'display_name' => 'Manage Warehouse Settings',
                'description' => 'Can modify warehouse settings and configuration',
                'group' => 'warehouse_settings'
            ],

            // User Management
            [
                'name' => 'warehouse.users.view',
                'display_name' => 'View Users',
                'description' => 'Can view warehouse users',
                'group' => 'warehouse_users'
            ],
            [
                'name' => 'warehouse.users.create',
                'display_name' => 'Create Users',
                'description' => 'Can create new warehouse users',
                'group' => 'warehouse_users'
            ],
            [
                'name' => 'warehouse.users.edit',
                'display_name' => 'Edit Users',
                'description' => 'Can edit warehouse user details',
                'group' => 'warehouse_users'
            ],
            [
                'name' => 'warehouse.users.delete',
                'display_name' => 'Delete Users',
                'description' => 'Can delete warehouse users',
                'group' => 'warehouse_users'
            ],
            [
                'name' => 'warehouse.users.roles',
                'display_name' => 'Manage User Roles',
                'description' => 'Can assign and manage user roles',
                'group' => 'warehouse_users'
            ],

            // Activity Log
            [
                'name' => 'warehouse.activity-log.view',
                'display_name' => 'View Activity Log',
                'description' => 'Can view warehouse activity logs',
                'group' => 'warehouse_activity'
            ],

            // Locations Management
            [
                'name' => 'warehouse.locations.view',
                'display_name' => 'View Locations',
                'description' => 'Can view warehouse locations',
                'group' => 'warehouse_locations'
            ],
            [
                'name' => 'warehouse.locations.create',
                'display_name' => 'Create Locations',
                'description' => 'Can create new warehouse locations',
                'group' => 'warehouse_locations'
            ],
            [
                'name' => 'warehouse.locations.edit',
                'display_name' => 'Edit Locations',
                'description' => 'Can edit warehouse location details',
                'group' => 'warehouse_locations'
            ],
            [
                'name' => 'warehouse.locations.delete',
                'display_name' => 'Delete Locations',
                'description' => 'Can delete warehouse locations',
                'group' => 'warehouse_locations'
            ],

            // Physical Count
            [
                'name' => 'warehouse.inventory.count',
                'display_name' => 'Physical Inventory Count',
                'description' => 'Can perform physical inventory counts',
                'group' => 'warehouse_inventory'
            ],
        ];

        // Insert permissions
        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                $permission
            );
        }

        // Warehouse-specific roles
        $roles = [
            [
                'name' => 'warehouse_manager',
                'display_name' => 'Warehouse Manager',
                'description' => 'Full access to warehouse operations including approvals and settings',
                'is_active' => true,
                'permissions' => [
                    // Full dashboard access
                    'warehouse.dashboard.view',

                    // Full supplier management
                    'warehouse.suppliers.view',
                    'warehouse.suppliers.create',
                    'warehouse.suppliers.edit',
                    'warehouse.suppliers.delete',

                    // Full project management
                    'warehouse.projects.view',
                    'warehouse.projects.create',
                    'warehouse.projects.edit',
                    'warehouse.projects.delete',

                    // Full items management
                    'warehouse.items.view',
                    'warehouse.items.create',
                    'warehouse.items.edit',
                    'warehouse.items.delete',

                    // Full inventory management
                    'warehouse.inventory.view',
                    'warehouse.inventory.adjust',
                    'warehouse.inventory.transfer',

                    // Full incoming transactions
                    'warehouse.incoming.view',
                    'warehouse.incoming.create',
                    'warehouse.incoming.edit',
                    'warehouse.incoming.complete',

                    // Full outgoing transactions
                    'warehouse.outgoing.view',
                    'warehouse.outgoing.create',
                    'warehouse.outgoing.approve',
                    'warehouse.outgoing.fulfill',

                    // Full returns management
                    'warehouse.returns.view',
                    'warehouse.returns.create',
                    'warehouse.returns.process',

                    // Full reports access
                    'warehouse.reports.view',
                    'warehouse.reports.export',
                    'warehouse.reports.advanced',

                    // Full settings access
                    'warehouse.settings.view',
                    'warehouse.settings.manage',

                    // Full user management
                    'warehouse.users.view',
                    'warehouse.users.create',
                    'warehouse.users.edit',
                    'warehouse.users.delete',
                    'warehouse.users.roles',

                    // Activity log
                    'warehouse.activity-log.view',

                    // Locations management
                    'warehouse.locations.view',
                    'warehouse.locations.create',
                    'warehouse.locations.edit',
                    'warehouse.locations.delete',

                    // Physical count
                    'warehouse.inventory.count',
                ]
            ],
            [
                'name' => 'warehouse_supervisor',
                'display_name' => 'Warehouse Supervisor',
                'description' => 'Can manage daily operations and approve transactions but limited settings access',
                'is_active' => true,
                'permissions' => [
                    // Dashboard access
                    'warehouse.dashboard.view',

                    // View and edit suppliers
                    'warehouse.suppliers.view',
                    'warehouse.suppliers.create',
                    'warehouse.suppliers.edit',

                    // View and edit projects
                    'warehouse.projects.view',
                    'warehouse.projects.create',
                    'warehouse.projects.edit',

                    // Full items management
                    'warehouse.items.view',
                    'warehouse.items.create',
                    'warehouse.items.edit',

                    // Full inventory management
                    'warehouse.inventory.view',
                    'warehouse.inventory.adjust',
                    'warehouse.inventory.transfer',

                    // Full incoming transactions
                    'warehouse.incoming.view',
                    'warehouse.incoming.create',
                    'warehouse.incoming.edit',
                    'warehouse.incoming.complete',

                    // Full outgoing transactions
                    'warehouse.outgoing.view',
                    'warehouse.outgoing.create',
                    'warehouse.outgoing.approve',
                    'warehouse.outgoing.fulfill',

                    // Full returns management
                    'warehouse.returns.view',
                    'warehouse.returns.create',
                    'warehouse.returns.process',

                    // Reports access
                    'warehouse.reports.view',
                    'warehouse.reports.export',

                    // View settings only
                    'warehouse.settings.view',

                    // Limited user management
                    'warehouse.users.view',
                    'warehouse.users.edit',

                    // Activity log
                    'warehouse.activity-log.view',

                    // Locations management
                    'warehouse.locations.view',
                    'warehouse.locations.create',
                    'warehouse.locations.edit',

                    // Physical count
                    'warehouse.inventory.count',
                ]
            ],
            [
                'name' => 'warehouse_clerk',
                'display_name' => 'Warehouse Clerk',
                'description' => 'Can handle daily transactions and inventory updates',
                'is_active' => true,
                'permissions' => [
                    // Dashboard access
                    'warehouse.dashboard.view',

                    // View suppliers and projects
                    'warehouse.suppliers.view',
                    'warehouse.projects.view',

                    // View and create items
                    'warehouse.items.view',
                    'warehouse.items.create',
                    'warehouse.items.edit',

                    // Full inventory access
                    'warehouse.inventory.view',
                    'warehouse.inventory.adjust',

                    // Create and edit incoming transactions
                    'warehouse.incoming.view',
                    'warehouse.incoming.create',
                    'warehouse.incoming.edit',
                    'warehouse.incoming.complete',

                    // Create outgoing transactions
                    'warehouse.outgoing.view',
                    'warehouse.outgoing.create',
                    'warehouse.outgoing.fulfill',

                    // Handle returns
                    'warehouse.returns.view',
                    'warehouse.returns.create',
                    'warehouse.returns.process',

                    // Basic reports
                    'warehouse.reports.view',
                    'warehouse.reports.export',
                ]
            ],
            [
                'name' => 'project_manager',
                'display_name' => 'Project Manager',
                'description' => 'Can request materials for projects and view consumption reports',
                'is_active' => true,
                'permissions' => [
                    // Limited dashboard access
                    'warehouse.dashboard.view',

                    // View suppliers and projects
                    'warehouse.suppliers.view',
                    'warehouse.projects.view',

                    // View items
                    'warehouse.items.view',

                    // View inventory
                    'warehouse.inventory.view',

                    // Create material requests
                    'warehouse.outgoing.view',
                    'warehouse.outgoing.create',

                    // Create returns
                    'warehouse.returns.view',
                    'warehouse.returns.create',

                    // View reports
                    'warehouse.reports.view',
                ]
            ],
            [
                'name' => 'warehouse_viewer',
                'display_name' => 'Warehouse Viewer/Auditor',
                'description' => 'Read-only access to warehouse data for auditing and reporting',
                'is_active' => true,
                'permissions' => [
                    // Dashboard access
                    'warehouse.dashboard.view',

                    // View all master data
                    'warehouse.suppliers.view',
                    'warehouse.projects.view',
                    'warehouse.items.view',
                    'warehouse.inventory.view',

                    // View all transactions
                    'warehouse.incoming.view',
                    'warehouse.outgoing.view',
                    'warehouse.returns.view',

                    // Full reports access
                    'warehouse.reports.view',
                    'warehouse.reports.export',
                    'warehouse.reports.advanced',

                    // View settings
                    'warehouse.settings.view',
                ]
            ],
        ];

        // Create roles and assign permissions
        foreach ($roles as $roleData) {
            $permissions = $roleData['permissions'];
            unset($roleData['permissions']);

            $role = Role::firstOrCreate(
                ['name' => $roleData['name']],
                $roleData
            );

            // Get permission IDs
            $permissionIds = Permission::whereIn('name', $permissions)->pluck('id')->toArray();

            // Sync permissions to role
            $role->permissions()->sync($permissionIds);
        }

        $this->command->info('Warehouse roles and permissions have been created successfully!');
    }
}