<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\ProductCategory;
use App\Models\ProductStandard;
use App\Models\ProductApplication;
use App\Models\Product;
use App\Models\ProductSpecification;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class FloatGlassProductSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Clear existing data
        $this->command->info('Clearing existing products data...');
        ProductSpecification::truncate();
        Product::truncate();
        ProductApplication::truncate();
        ProductStandard::truncate();
        ProductCategory::truncate();

        // Copy images to storage
        $this->copyImagesToStorage();

        // Create Float Glass Category
        $this->command->info('Creating Float Glass category...');
        $floatGlassCategory = ProductCategory::create([
            'name' => 'Float Glass',
            'slug' => 'float-glass',
            'description' => 'High-quality float glass products manufactured to various international standards',
            'image' => 'product-categories/float-glass.jpg',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // Create Standards
        $this->command->info('Creating standards...');
        
        // JIS Standard
        $jisStandard = ProductStandard::create([
            'category_id' => $floatGlassCategory->id,
            'name' => 'Japanese Industrial Standard (JIS)',
            'code' => 'JIS R 3202',
            'description' => 'Japanese Industrial Standard for flat glass specifications',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // EN Standard
        $enStandard = ProductStandard::create([
            'category_id' => $floatGlassCategory->id,
            'name' => 'European Standard (EN)',
            'code' => 'EN 572-2',
            'description' => 'European standard for float glass specifications',
            'sort_order' => 2,
            'is_active' => true,
        ]);

        // ASTM Standard
        $astmStandard = ProductStandard::create([
            'category_id' => $floatGlassCategory->id,
            'name' => 'American Standards (ASTM)',
            'code' => 'ASTM C1036',
            'description' => 'American Society for Testing and Materials standard for flat glass',
            'sort_order' => 3,
            'is_active' => true,
        ]);

        // SASO Standard
        $sasoStandard = ProductStandard::create([
            'category_id' => $floatGlassCategory->id,
            'name' => 'Saudi Standards Organization (SASO)',
            'code' => 'SASO 2663',
            'description' => 'Saudi Standards Organization specifications for glass products',
            'sort_order' => 4,
            'is_active' => true,
        ]);

        // Create Applications
        $this->command->info('Creating applications...');
        
        // JIS Architecture Application
        $jisArchitecture = ProductApplication::create([
            'standard_id' => $jisStandard->id,
            'name' => 'Architecture',
            'description' => 'Architectural glass applications meeting JIS standards',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // EN Architecture Application
        $enArchitecture = ProductApplication::create([
            'standard_id' => $enStandard->id,
            'name' => 'Architecture',
            'description' => 'Architectural glass applications meeting European standards',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // ASTM Automotive Application
        $astmAutomotive = ProductApplication::create([
            'standard_id' => $astmStandard->id,
            'name' => 'Automotive',
            'description' => 'Automotive glass applications meeting ASTM standards',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // SASO Architecture Application
        $sasoArchitecture = ProductApplication::create([
            'standard_id' => $sasoStandard->id,
            'name' => 'SASO Architecture',
            'description' => 'Architectural glass applications meeting SASO standards',
            'sort_order' => 1,
            'is_active' => true,
        ]);

        // Create Products
        $this->command->info('Creating products...');

        // JIS Architecture Products
        $this->createArchitectureProducts($jisArchitecture, 'JIS');

        // EN Architecture Products
        $this->createArchitectureProducts($enArchitecture, 'EN');

        // ASTM Automotive Products
        $this->createAutomotiveProducts($astmAutomotive);

        // SASO Architecture Products
        $this->createArchitectureProducts($sasoArchitecture, 'SASO');

        $this->command->info('Float Glass product seeding completed successfully!');
    }

    private function copyImagesToStorage()
    {
        $this->command->info('Copying images to storage...');

        // Ensure storage directories exist
        Storage::disk('public')->makeDirectory('product-categories');
        Storage::disk('public')->makeDirectory('products');

        // Copy Float Glass images
        $sourceDir = base_path('html/Images/Float');
        $targetDir = storage_path('app/public/products');

        if (File::exists($sourceDir)) {
            $files = File::files($sourceDir);
            foreach ($files as $file) {
                $filename = $file->getFilename();
                File::copy($file->getPathname(), $targetDir . '/' . $filename);
            }
        }

        // Copy a sample image for category
        $categoryImageSource = base_path('html/Images/Float/DSC00418.jpg');
        $categoryImageTarget = storage_path('app/public/product-categories/float-glass.jpg');
        
        if (File::exists($categoryImageSource)) {
            File::copy($categoryImageSource, $categoryImageTarget);
        }
    }

    private function createArchitectureProducts($application, $standardType)
    {
        $thicknesses = ['3mm', '4mm', '5mm', '5.5mm', '6mm', '8mm', '10mm', '12mm'];
        $images = ['DSC00418.jpg', 'DSC00415.jpg', 'WhatsApp Image 2025-07-16 at 2.59.04 PM.jpeg'];
        
        foreach ($thicknesses as $index => $thickness) {
            $imageFile = $images[$index % count($images)];
            
            $product = Product::create([
                'application_id' => $application->id,
                'name' => "Clear Float {$thickness}",
                'slug' => "clear-float-{$thickness}-{$standardType}",
                'description' => "High-quality clear float glass with {$thickness} thickness, manufactured according to {$standardType} standards for architectural applications.",
                'features' => "Premium clarity, excellent optical properties, uniform thickness, smooth surface finish",
                'image' => "products/{$imageFile}",
                'thickness_min' => (float) str_replace('mm', '', $thickness),
                'thickness_max' => (float) str_replace('mm', '', $thickness),
                'thickness_unit' => 'mm',
                'sort_order' => $index + 1,
                'is_featured' => in_array($thickness, ['5mm', '6mm']),
                'is_active' => true,
            ]);

            // Add specifications
            $this->addProductSpecifications($product, $thickness, $standardType);
        }
    }

    private function createAutomotiveProducts($application)
    {
        $products = [
            ['name' => 'Automotive Clear Float 4mm', 'thickness' => '4mm'],
            ['name' => 'Automotive Clear Float 2.5mm', 'thickness' => '2.5mm'],
        ];

        $images = ['DSC00418.jpg', 'DSC00415.jpg'];

        foreach ($products as $index => $productData) {
            $thickness = $productData['thickness'];
            $imageFile = $images[$index % count($images)];
            
            $product = Product::create([
                'application_id' => $application->id,
                'name' => $productData['name'],
                'slug' => str_replace(' ', '-', strtolower($productData['name'])),
                'description' => "Specialized automotive float glass with {$thickness} thickness, designed for vehicle glazing applications according to ASTM standards.",
                'features' => "Enhanced safety properties, optical clarity, precise thickness control, automotive grade quality",
                'image' => "products/{$imageFile}",
                'thickness_min' => (float) str_replace('mm', '', $thickness),
                'thickness_max' => (float) str_replace('mm', '', $thickness),
                'thickness_unit' => 'mm',
                'sort_order' => $index + 1,
                'is_featured' => true,
                'is_active' => true,
            ]);

            // Add automotive specifications
            $this->addAutomotiveSpecifications($product, $thickness);
        }
    }

    private function addProductSpecifications($product, $thickness, $standardType)
    {
        $thicknessValue = (float) str_replace('mm', '', $thickness);

        // Size specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'size',
            'name' => 'Maximum Size',
            'value' => '3210 x 2250',
            'unit' => 'mm',
            'sort_order' => 1,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'size',
            'name' => 'Minimum Size',
            'value' => '300 x 300',
            'unit' => 'mm',
            'sort_order' => 2,
        ]);

        // Technical specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'technical',
            'name' => 'Thickness Tolerance',
            'value' => $thicknessValue <= 6 ? '±0.2' : '±0.3',
            'unit' => 'mm',
            'sort_order' => 3,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'technical',
            'name' => 'Density',
            'value' => '2500',
            'unit' => 'kg/m³',
            'sort_order' => 4,
        ]);

        // Optical specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'optical',
            'name' => 'Light Transmission',
            'value' => '90',
            'unit' => '%',
            'sort_order' => 5,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'optical',
            'name' => 'Light Reflection',
            'value' => '8',
            'unit' => '%',
            'sort_order' => 6,
        ]);

        // Quality specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'quality',
            'name' => 'Standard Compliance',
            'value' => $standardType,
            'unit' => null,
            'sort_order' => 7,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'quality',
            'name' => 'Surface Quality',
            'value' => 'Premium Grade',
            'unit' => null,
            'sort_order' => 8,
        ]);
    }

    private function addAutomotiveSpecifications($product, $thickness)
    {
        $thicknessValue = (float) str_replace('mm', '', $thickness);

        // Automotive size specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'size',
            'name' => 'Maximum Size',
            'value' => '2440 x 1830',
            'unit' => 'mm',
            'sort_order' => 1,
        ]);

        // Technical specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'technical',
            'name' => 'Thickness Tolerance',
            'value' => '±0.1',
            'unit' => 'mm',
            'sort_order' => 2,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'technical',
            'name' => 'Tensile Strength',
            'value' => '50',
            'unit' => 'MPa',
            'sort_order' => 3,
        ]);

        // Safety specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'quality',
            'name' => 'Safety Standard',
            'value' => 'ANSI Z26.1',
            'unit' => null,
            'sort_order' => 4,
        ]);

        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'quality',
            'name' => 'Application Grade',
            'value' => 'Automotive Grade',
            'unit' => null,
            'sort_order' => 5,
        ]);

        // Optical specifications
        ProductSpecification::create([
            'product_id' => $product->id,
            'spec_type' => 'optical',
            'name' => 'Light Transmission',
            'value' => '91',
            'unit' => '%',
            'sort_order' => 6,
        ]);
    }
}