<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('damaged_items', function (Blueprint $table) {
            $table->id();

            // Item details
            $table->foreignId('item_id')->constrained('items')->onDelete('cascade');
            $table->decimal('quantity', 15, 4);
            $table->string('unit')->nullable();
            $table->decimal('unit_cost', 15, 4)->default(0);
            $table->decimal('total_cost', 15, 4)->default(0);

            // Source information
            $table->enum('source_type', ['site_return', 'inventory_conversion'])->index();
            $table->string('source_reference')->nullable(); // Reference to original transaction

            // Project details (for site returns)
            $table->foreignId('project_id')->nullable()->constrained('projects')->onDelete('set null');
            $table->text('project_details')->nullable();

            // Inventory details (for inventory conversions)
            $table->foreignId('inventory_id')->nullable()->constrained('inventory')->onDelete('set null');
            $table->text('inventory_reference')->nullable();

            // Damage information
            $table->text('damage_description');
            $table->enum('damage_severity', ['minor', 'moderate', 'major', 'total_loss'])->default('moderate');
            $table->text('damage_notes')->nullable();

            // Cost responsibility and status
            $table->enum('cost_responsibility', ['project', 'company', 'supplier', 'insurance', 'pending'])->default('pending');
            $table->enum('status', ['reported', 'investigating', 'approved', 'settled', 'written_off'])->default('reported');
            $table->decimal('recoverable_amount', 15, 4)->default(0);
            $table->decimal('loss_amount', 15, 4)->default(0);

            // Approval and tracking
            $table->foreignId('reported_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('approved_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamp('reported_at');
            $table->timestamp('approved_at')->nullable();

            // Additional metadata
            $table->json('attachments')->nullable(); // Photos, documents
            $table->text('resolution_notes')->nullable();
            $table->enum('disposal_method', ['repair', 'salvage', 'scrap', 'return_supplier', 'pending'])->nullable();
            $table->timestamp('disposed_at')->nullable();

            $table->timestamps();
            $table->softDeletes();

            // Indexes for better performance
            $table->index(['source_type', 'status']);
            $table->index(['project_id', 'status']);
            $table->index(['cost_responsibility']);
            $table->index(['reported_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('damaged_items');
    }
};
