<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('direct_deliveries', function (Blueprint $table) {
            // Link to Material Request
            $table->foreignId('material_request_id')
                ->nullable()
                ->after('id')
                ->constrained('material_transfer_requests')
                ->onDelete('set null');

            // Operation Type for different delivery scenarios
            $table->enum('operation_type', ['supplier_delivery', 'site_return', 'inter_division_transfer', 'emergency_procurement'])
                ->default('supplier_delivery')
                ->after('delivery_type');

            // Project and Division Details
            $table->string('project_phase')->nullable()->after('project_division_id');
            $table->string('site_location')->nullable()->after('project_phase');
            $table->json('gps_coordinates')->nullable()->after('site_location');

            // LPO Management Fields
            $table->string('lpo_number')->nullable()->after('purchase_order_number');
            $table->enum('lpo_status', ['pending', 'approved', 'partially_delivered', 'completed', 'cancelled'])
                ->nullable()
                ->after('lpo_number');
            $table->decimal('lpo_budget_amount', 15, 2)->nullable()->after('lpo_status');
            $table->decimal('exchange_rate', 10, 4)->default(1.0000)->after('currency');

            // Enhanced Delivery Information
            $table->string('driver_contact')->nullable()->after('vehicle_number');
            $table->text('delivery_instructions')->nullable()->after('delivery_notes');

            // Quality Control Fields
            $table->boolean('requires_inspection')->default(true)->after('requires_approval');
            $table->foreignId('inspector_id')
                ->nullable()
                ->after('requires_inspection')
                ->constrained('users')
                ->nullOnDelete();
            $table->dateTime('inspection_datetime')->nullable()->after('inspector_id');
            $table->enum('inspection_result', ['passed', 'failed', 'conditional', 'pending'])
                ->nullable()
                ->after('inspection_datetime');
            $table->text('inspection_notes')->nullable()->after('inspection_result');
            $table->json('inspection_checklist')->nullable()->after('inspection_notes');

            // VAT and Financial Information
            $table->decimal('vat_amount', 15, 2)->default(0)->after('discount_amount');
            $table->decimal('vat_rate', 5, 2)->default(5.00)->after('vat_amount');
            $table->decimal('total_amount_aed', 15, 2)->nullable()->after('total_amount');

            // Compliance & Audit
            $table->boolean('regulatory_compliant')->default(false)->after('can_be_returned');
            $table->json('compliance_checklist')->nullable()->after('regulatory_compliant');

            // Documents
            $table->json('attached_documents')->nullable()->after('receipt_document_path');
            $table->json('photo_documentation')->nullable()->after('attached_documents');

            // Workflow Status
            $table->enum('workflow_status', ['draft', 'pending_inspection', 'approved', 'rejected', 'completed'])
                ->default('draft')
                ->after('status');

            // Additional Person Information
            $table->string('received_by_name')->nullable()->after('received_by');

            // Indexes for performance
            $table->index(['material_request_id']);
            $table->index(['operation_type']);
            $table->index(['lpo_number']);
            $table->index(['workflow_status']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('direct_deliveries', function (Blueprint $table) {
            // Drop indexes
            $table->dropIndex(['material_request_id']);
            $table->dropIndex(['operation_type']);
            $table->dropIndex(['lpo_number']);
            $table->dropIndex(['workflow_status']);

            // Drop foreign keys
            $table->dropForeign(['material_request_id']);
            $table->dropForeign(['inspector_id']);

            // Drop columns
            $table->dropColumn([
                'material_request_id',
                'operation_type',
                'project_phase',
                'site_location',
                'gps_coordinates',
                'lpo_number',
                'lpo_status',
                'lpo_budget_amount',
                'exchange_rate',
                'driver_contact',
                'delivery_instructions',
                'requires_inspection',
                'inspector_id',
                'inspection_datetime',
                'inspection_result',
                'inspection_notes',
                'inspection_checklist',
                'vat_amount',
                'vat_rate',
                'total_amount_aed',
                'regulatory_compliant',
                'compliance_checklist',
                'attached_documents',
                'photo_documentation',
                'workflow_status',
                'received_by_name'
            ]);
        });
    }
};