<?php

require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\IncomingOperation;
use App\Models\DirectDelivery;

try {
    echo "=== Comprehensive Direct Delivery Inventory Fix ===" . PHP_EOL;
    echo "Properly syncing DD-202511-0006 with project inventory..." . PHP_EOL . PHP_EOL;

    DB::beginTransaction();

    // Step 1: Reset project inventory for laptops to known state
    echo "Step 1: Resetting laptop inventory for project 15..." . PHP_EOL;

    $existingLaptopInventory = ProjectInventory::where('project_id', 15)
        ->where('item_id', 1)
        ->first();

    if ($existingLaptopInventory) {
        $existingLaptopInventory->delete();
        echo "  Deleted existing laptop inventory record" . PHP_EOL;
    }

    // Step 2: Create initial project inventory (simulate the 500 laptops from other sources)
    echo PHP_EOL . "Step 2: Creating initial laptop inventory (500 units from previous deliveries)..." . PHP_EOL;

    ProjectInventory::create([
        'project_id' => 15,
        'item_id' => 1, // laptop
        'quantity_available' => 500,
        'allocated_quantity' => 0,
        'unit_price' => 1500.00, // Default price
        'total_value' => 500 * 1500.00
    ]);

    echo "  Created initial inventory: 500 laptops" . PHP_EOL;

    // Step 3: Process the direct delivery to add to project inventory
    echo PHP_EOL . "Step 3: Processing DD-202511-0006 delivery to project inventory..." . PHP_EOL;

    $dd = DirectDelivery::where('delivery_number', 'DD-202511-0006')->first();
    $laptopDeliveryItem = $dd->items()->where('item_id', 1)->first();

    if ($laptopDeliveryItem && $laptopDeliveryItem->received_quantity > 0) {
        $projectInventory = ProjectInventory::where('project_id', 15)
            ->where('item_id', 1)
            ->first();

        if ($projectInventory) {
            $oldQuantity = $projectInventory->quantity_available;
            $projectInventory->updateQuantity($laptopDeliveryItem->received_quantity, 'add');
            $newQuantity = $projectInventory->quantity_available;

            echo "  Added delivery: {$oldQuantity} + {$laptopDeliveryItem->received_quantity} = {$newQuantity}" . PHP_EOL;
        }
    }

    // Step 4: Process site returns to subtract from project inventory
    echo PHP_EOL . "Step 4: Processing site returns (only up to available quantity)..." . PHP_EOL;

    $returnIds = [11, 12, 15, 16, 17, 18];
    $totalSubtracted = 0;

    foreach ($returnIds as $id) {
        $return = IncomingOperation::find($id);
        if ($return) {
            foreach ($return->items as $item) {
                if ($item->item_id == 1) { // laptop
                    $projectInventory = ProjectInventory::where('project_id', 15)
                        ->where('item_id', 1)
                        ->first();

                    if ($projectInventory && $projectInventory->quantity_available > 0) {
                        $quantityToSubtract = min($item->quantity_accepted, $projectInventory->quantity_available);

                        $oldQuantity = $projectInventory->quantity_available;
                        $projectInventory->updateQuantity($quantityToSubtract, 'subtract');
                        $newQuantity = $projectInventory->quantity_available;

                        echo "  Return {$id}: {$oldQuantity} - {$quantityToSubtract} = {$newQuantity}" . PHP_EOL;
                        $totalSubtracted += $quantityToSubtract;

                        if ($projectInventory->quantity_available <= 0) {
                            $projectInventory->delete();
                            echo "    Deleted inventory record (quantity = 0)" . PHP_EOL;
                            break; // No more inventory to subtract from
                        }
                    } else {
                        echo "  Return {$id}: Skipped (no inventory available)" . PHP_EOL;
                    }
                }
            }
        }
    }

    DB::commit();

    echo PHP_EOL . "=== Summary ===" . PHP_EOL;
    echo "Initial inventory: 500 laptops" . PHP_EOL;
    echo "Added from DD-202511-0006: 300 laptops" . PHP_EOL;
    echo "Total available: 800 laptops" . PHP_EOL;
    echo "Subtracted from returns: {$totalSubtracted} laptops" . PHP_EOL;
    echo "Final inventory: " . (800 - $totalSubtracted) . " laptops" . PHP_EOL;

    // Show final project inventory state
    echo PHP_EOL . "=== Final Project Inventory ===" . PHP_EOL;
    $finalInventory = ProjectInventory::where('project_id', 15)->with('item')->get();

    if ($finalInventory->count() > 0) {
        foreach ($finalInventory as $item) {
            echo "{$item->item->item_description}: {$item->quantity_available} units" . PHP_EOL;
        }
    } else {
        echo "No items remaining in project inventory" . PHP_EOL;
    }

} catch (Exception $e) {
    DB::rollback();
    echo "Error: " . $e->getMessage() . PHP_EOL;
    echo "Trace: " . $e->getTraceAsString() . PHP_EOL;
}