<?php

namespace App\Policies\Warehouse;

use App\Models\User;
use App\Models\Warehouse\Item;

class ItemPolicy extends BaseWarehousePolicy
{
    /**
     * Determine whether the user can view any items.
     */
    public function viewAny(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.view');
    }

    /**
     * Determine whether the user can view the item.
     */
    public function view(User $user, Item $item): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.view');
    }

    /**
     * Determine whether the user can create items.
     */
    public function create(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.create');
    }

    /**
     * Determine whether the user can update the item.
     */
    public function update(User $user, Item $item): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.edit');
    }

    /**
     * Determine whether the user can delete the item.
     */
    public function delete(User $user, Item $item): bool
    {
        // Only managers can delete items, and only if they have no stock
        if (!$this->hasWarehousePermission($user, 'warehouse.items.delete')) {
            return false;
        }

        // Check if item has stock
        if ($item->total_quantity > 0) {
            return false;
        }

        return $this->isWarehouseManagerOrHigher($user);
    }

    /**
     * Determine whether the user can restore the item.
     */
    public function restore(User $user, Item $item): bool
    {
        return $this->isWarehouseManagerOrHigher($user);
    }

    /**
     * Determine whether the user can permanently delete the item.
     */
    public function forceDelete(User $user, Item $item): bool
    {
        return $this->accessService->isSuperAdmin($user);
    }

    /**
     * Determine whether the user can bulk import items.
     */
    public function bulkImport(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.create') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can export items.
     */
    public function export(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.view');
    }

    /**
     * Determine whether the user can generate barcodes.
     */
    public function generateBarcode(User $user, Item $item): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.items.view');
    }
}