<?php

namespace App\Policies\Warehouse;

use App\Models\User;
use App\Models\Warehouse\Inventory;

class InventoryPolicy extends BaseWarehousePolicy
{
    /**
     * Determine whether the user can view any inventory.
     */
    public function viewAny(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.view');
    }

    /**
     * Determine whether the user can view the inventory.
     */
    public function view(User $user, $model): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.view');
    }

    /**
     * Determine whether the user can adjust inventory.
     */
    public function adjust(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.adjust') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can transfer inventory.
     */
    public function transfer(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.transfer') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can perform physical count.
     */
    public function physicalCount(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.count') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can view stock movements.
     */
    public function viewStockMovements(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.inventory.view');
    }

    /**
     * Determine whether the user can reserve inventory.
     */
    public function reserve(User $user): bool
    {
        return $this->hasAnyWarehousePermission($user, [
            'warehouse.outgoing.create',
            'warehouse.inventory.adjust'
        ]);
    }

    /**
     * Determine whether the user can release reserved inventory.
     */
    public function release(User $user): bool
    {
        return $this->hasAnyWarehousePermission($user, [
            'warehouse.outgoing.edit',
            'warehouse.inventory.adjust'
        ]);
    }
}