<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MaterialTransferRequestItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'transfer_request_id',
        'item_id',
        'outgoing_item_id',
        'direct_delivery_item_id',
        'material_transfer_item_id',
        'quantity_requested',
        'quantity_available',
        'quantity_approved',
        'unit_of_measure',
        'estimated_unit_price',
        'estimated_total_value',
        'item_specifications',
        'transfer_notes',
        'approval_status',
        'approval_notes'
    ];

    protected $casts = [
        'quantity_requested' => 'decimal:2',
        'quantity_available' => 'decimal:2',
        'quantity_approved' => 'decimal:2',
        'estimated_unit_price' => 'decimal:2',
        'estimated_total_value' => 'decimal:2'
    ];

    protected static function boot()
    {
        parent::boot();

        static::saving(function ($requestItem) {
            // Auto-calculate estimated total value
            if ($requestItem->quantity_requested && $requestItem->estimated_unit_price) {
                $requestItem->estimated_total_value = $requestItem->quantity_requested * $requestItem->estimated_unit_price;
            }
        });
    }

    /**
     * Relationships
     */
    public function transferRequest(): BelongsTo
    {
        return $this->belongsTo(MaterialTransferRequest::class, 'transfer_request_id');
    }

    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    public function outgoingItem(): BelongsTo
    {
        return $this->belongsTo(OutgoingItem::class, 'outgoing_item_id');
    }

    public function directDeliveryItem(): BelongsTo
    {
        return $this->belongsTo(\App\Models\DirectDeliveryItem::class, 'direct_delivery_item_id');
    }

    public function materialTransferItem(): BelongsTo
    {
        return $this->belongsTo(MaterialTransferRequestItem::class, 'material_transfer_item_id');
    }

    /**
     * Approval Management
     */
    public function approve($quantity = null, $notes = null)
    {
        $this->quantity_approved = $quantity ?? $this->quantity_requested;
        $this->approval_status = 'approved';
        $this->approval_notes = $notes;
        $this->save();
    }

    public function reject($notes = null)
    {
        $this->approval_status = 'rejected';
        $this->approval_notes = $notes;
        $this->save();
    }

    public function modify($quantity, $notes = null)
    {
        $this->quantity_approved = $quantity;
        $this->approval_status = 'modified';
        $this->approval_notes = $notes;
        $this->save();
    }

    /**
     * Status Checks
     */
    public function isPending()
    {
        return $this->approval_status === 'pending';
    }

    public function isApproved()
    {
        return in_array($this->approval_status, ['approved', 'modified']);
    }

    public function isRejected()
    {
        return $this->approval_status === 'rejected';
    }

    public function isModified()
    {
        return $this->approval_status === 'modified';
    }

    /**
     * Quantity Validation
     */
    public function hasAvailableQuantity()
    {
        return $this->quantity_available >= $this->quantity_requested;
    }

    public function getShortfallQuantityAttribute()
    {
        return max(0, $this->quantity_requested - $this->quantity_available);
    }

    /**
     * Get approval status badge color
     */
    public function getApprovalStatusColorAttribute()
    {
        return match($this->approval_status) {
            'pending' => 'warning',
            'approved' => 'success',
            'modified' => 'info',
            'rejected' => 'danger',
            default => 'secondary'
        };
    }
}