<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\FocInventory;
use App\Models\Warehouse\Item;
use App\Models\User;

class FocInventoryAdjustment extends Model
{
    use HasFactory;

    protected $fillable = [
        'foc_inventory_id',
        'item_id',
        'adjustment_type',
        'quantity_before',
        'quantity_after',
        'quantity_change',
        'condition_before',
        'condition_after',
        'status_before',
        'status_after',
        'reason',
        'notes',
        'reference_type',
        'reference_id',
        'reference_number',
        'cost_impact',
        'approval_status',
        'requested_by',
        'approved_by',
        'approved_at',
    ];

    protected $casts = [
        'quantity_before' => 'decimal:4',
        'quantity_after' => 'decimal:4',
        'quantity_change' => 'decimal:4',
        'cost_impact' => 'decimal:2',
        'approved_at' => 'datetime',
    ];

    // Relationships
    public function focInventory()
    {
        return $this->belongsTo(FocInventory::class);
    }

    public function item()
    {
        return $this->belongsTo(Item::class);
    }

    public function requestedBy()
    {
        return $this->belongsTo(User::class, 'requested_by');
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // Scopes
    public function scopeApproved($query)
    {
        return $query->where('approval_status', 'approved');
    }

    public function scopePending($query)
    {
        return $query->where('approval_status', 'pending');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('adjustment_type', $type);
    }

    public function scopeForItem($query, $itemId)
    {
        return $query->where('item_id', $itemId);
    }

    public function scopeWithReference($query, $type, $id)
    {
        return $query->where('reference_type', $type)
                    ->where('reference_id', $id);
    }

    // Helper methods
    public function approve($userId = null)
    {
        $this->approval_status = 'approved';
        $this->approved_by = $userId ?? auth()->id();
        $this->approved_at = now();
        $this->save();

        return $this;
    }

    public function reject($userId = null)
    {
        $this->approval_status = 'rejected';
        $this->approved_by = $userId ?? auth()->id();
        $this->approved_at = now();
        $this->save();

        return $this;
    }

    public function getFormattedChangeAttribute()
    {
        $change = $this->quantity_change;
        return $change >= 0 ? '+' . number_format($change, 4) : number_format($change, 4);
    }

    public function getImpactColorAttribute()
    {
        if ($this->quantity_change > 0) return 'success';
        if ($this->quantity_change < 0) return 'danger';
        return 'info';
    }
}