<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Services\ExternalProjectImportService;
use Illuminate\Http\Request;

class ProjectController extends Controller
{
    public function __construct()
    {
        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.projects.view')->only(['index', 'show']);
        $this->middleware('warehouse.permission:warehouse.projects.create')->only(['create', 'store']);
        $this->middleware('warehouse.permission:warehouse.projects.edit')->only(['edit', 'update']);
        $this->middleware('warehouse.permission:warehouse.projects.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        if ($request->ajax() || $request->expectsJson()) {
            return $this->getDataForTable($request);
        }

        return view('warehouse.projects.index');
    }

    public function create()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.create
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        return view('warehouse.projects.create', compact('divisions'));
    }

    public function store(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.create

        $validatedData = $request->validate([
            'project_number' => 'nullable|string|max:50|unique:projects,project_number',
            'project_name' => 'required|string|max:255|unique:projects,project_name',
            'project_division_id' => 'required|exists:project_divisions,id',
        ]);

        // Auto-generate project number if not provided
        if (empty($validatedData['project_number'])) {
            $validatedData['project_number'] = Project::generateProjectNumber($validatedData['project_division_id']);
        }

        try {
            $project = Project::create($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $project,
                    'message' => 'Project created successfully'
                ], 201);
            }

            return redirect()->route('warehouse.projects.index')
                           ->with('success', 'Project created successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    public function show($id)
    {
        $project = Project::with('projectDivision')->findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.view

        return view('warehouse.projects.show', compact('project'));
    }

    public function edit($id)
    {
        $project = Project::with('projectDivision')->findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.edit
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();

        return view('warehouse.projects.edit', compact('project', 'divisions'));
    }

    public function update(Request $request, $id)
    {
        $project = Project::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.edit

        $validatedData = $request->validate([
            'project_number' => 'nullable|string|max:50|unique:projects,project_number,' . $id,
            'project_name' => 'required|string|max:255|unique:projects,project_name,' . $id,
            'project_division_id' => 'required|exists:project_divisions,id',
        ]);

        // Auto-generate project number if not provided and project doesn't have one
        if (empty($validatedData['project_number']) && empty($project->project_number)) {
            $validatedData['project_number'] = Project::generateProjectNumber($validatedData['project_division_id']);
        }

        try {
            $project->update($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $project,
                    'message' => 'Project updated successfully'
                ]);
            }

            return redirect()->route('warehouse.projects.index')
                           ->with('success', 'Project updated successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    public function destroy($id)
    {
        $project = Project::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.projects.delete

        try {
            $project->delete();

            return response()->json([
                'success' => true,
                'message' => 'Project deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Get data for DataTables.
     */
    protected function getDataForTable(Request $request)
    {
        try {
            $search = $request->input('search.value');

            // Get total count before any filtering
            $totalRecords = Project::count();

            $query = Project::with('projectDivision');

            if (!empty($search)) {
                $query->where(function($q) use ($search) {
                    $q->where('project_name', 'LIKE', "%{$search}%")
                      ->orWhere('project_number', 'LIKE', "%{$search}%")
                      ->orWhere('project_id', 'LIKE', "%{$search}%")
                      ->orWhereHas('projectDivision', function($q) use ($search) {
                          $q->where('division_name', 'LIKE', "%{$search}%");
                      });
                });
            }

            $filteredRecords = $query->count();

            // Apply ordering
            if ($request->has('order')) {
                $orderColumn = $request->input('columns.' . $request->input('order.0.column') . '.data');
                $orderDirection = $request->input('order.0.dir');

                if ($orderColumn === 'project_name') {
                    $query->orderBy('project_name', $orderDirection);
                } elseif ($orderColumn === 'project_id') {
                    $query->orderBy('project_id', $orderDirection);
                } elseif ($orderColumn === 'project_number') {
                    $query->orderBy('project_number', $orderDirection);
                } else {
                    $query->orderBy('created_at', $orderDirection);
                }
            } else {
                $query->orderBy('created_at', 'desc');
            }

            // Apply pagination
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);

            if ($length != -1) {
                $query->skip($start)->take($length);
            }

            $projects = $query->get();

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $projects->map(function ($project) {
                    return [
                        'id' => $project->id,
                        'project_number' => $project->project_number ?: '-',
                        'project_id' => $project->project_id ?: '-',
                        'project_name' => $project->project_name,
                        'division' => $project->projectDivision ? $project->projectDivision->division_name : '-',
                        'created_at' => $project->created_at->format('M d, Y'),
                        'actions' => $this->getActionButtons($project),
                    ];
                })->toArray()
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => []
            ], 400);
        }
    }

    private function getActionButtons($project)
    {
        $buttons = [];

        $buttons[] = '<a href="' . route('warehouse.projects.show', $project->id) . '" class="btn btn-sm btn-info" title="View"><i class="fa fa-eye"></i></a>';

        if (auth()->user()->can('warehouse.projects.edit')) {
            $buttons[] = '<a href="' . route('warehouse.projects.edit', $project->id) . '" class="btn btn-sm btn-primary" title="Edit"><i class="fa fa-edit"></i></a>';
        }

        if (auth()->user()->can('warehouse.projects.delete')) {
            $buttons[] = '<button onclick="deleteProject(' . $project->id . ')" class="btn btn-sm btn-danger" title="Delete"><i class="fa fa-trash"></i></button>';
        }

        return implode(' ', $buttons);
    }

    public function syncExternal(Request $request)
    {
        try {
            $importService = new ExternalProjectImportService();

            $options = [
                'table_name' => 'project',
                'batch_size' => $request->input('batch_size', 50),
                'skip_existing' => $request->input('skip_existing', false), // Changed default to false to allow updates
                'update_existing' => $request->input('update_existing', true), // NEW: Update existing records
                'dry_run' => $request->input('dry_run', false),
            ];

            $results = $importService->importProjects($options);

            // Build response message with division and project sync details
            $message = $results['dry_run'] ?? false
                ? 'Data validation completed'
                : 'Data sync completed successfully';

            // Add division sync summary
            $divisionSync = $results['division_sync'] ?? [];
            if (!empty($divisionSync)) {
                $divisionSummary = [];
                if (($divisionSync['created'] ?? 0) > 0) {
                    $divisionSummary[] = "{$divisionSync['created']} divisions created";
                }
                if (($divisionSync['updated'] ?? 0) > 0) {
                    $divisionSummary[] = "{$divisionSync['updated']} divisions updated";
                }
                if (!empty($divisionSummary)) {
                    $message .= '. Divisions: ' . implode(', ', $divisionSummary);
                }
            }

            // Add project sync summary
            $projectSummary = [];
            if (($results['imported'] ?? 0) > 0) {
                $projectSummary[] = "{$results['imported']} projects created";
            }
            if (($results['updated'] ?? 0) > 0) {
                $projectSummary[] = "{$results['updated']} projects updated";
            }
            if (!empty($projectSummary)) {
                $message .= '. Projects: ' . implode(', ', $projectSummary);
            }

            if ($results['validation_errors'] > 0) {
                $message .= " with {$results['validation_errors']} validation warnings";
            }

            // Add skip reasons if projects were skipped
            if ($results['skipped'] > 0 && !empty($results['error_details'])) {
                $results['skip_reasons'] = array_slice($results['error_details'], 0, 10); // First 10 for display
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Sync failed: ' . $e->getMessage()
            ], 400);
        }
    }

    public function inspectExternal(Request $request)
    {
        try {
            $importService = new ExternalProjectImportService();
            $inspection = $importService->inspectExternalTables();

            return response()->json([
                'success' => true,
                'data' => $inspection
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Inspection failed: ' . $e->getMessage()
            ], 400);
        }
    }

    public function getSyncStatistics(Request $request)
    {
        try {
            $importService = new ExternalProjectImportService();
            $stats = $importService->getExternalIdStatistics();

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get statistics: ' . $e->getMessage()
            ], 400);
        }
    }

    public function backfillExternalIds(Request $request)
    {
        try {
            $importService = new ExternalProjectImportService();

            $options = [
                'batch_size' => $request->input('batch_size', 50)
            ];

            $results = $importService->backfillExternalIds($options);

            return response()->json([
                'success' => true,
                'message' => 'External ID backfill completed',
                'data' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Backfill failed: ' . $e->getMessage()
            ], 400);
        }
    }
}