<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\MaterialSiteTransfer;
use App\Models\Warehouse\MaterialSiteTransferItem;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
// use Yajra\DataTables\DataTables;

class MaterialSiteTransferController extends Controller
{
    /**
     * Display a listing of site transfers.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            try {
                $query = MaterialSiteTransfer::with([
                    'fromProject',
                    'toProject',
                    'transferredBy'
                ])->latest('transfer_date');

                // Apply filters
                if ($request->filled('status')) {
                    $query->where('status', $request->status);
                }

                if ($request->filled('date_from')) {
                    $query->where('transfer_date', '>=', $request->date_from);
                }

                if ($request->filled('date_to')) {
                    $query->where('transfer_date', '<=', $request->date_to);
                }

                // Get pagination parameters
                $start = $request->get('start', 0);
                $length = $request->get('length', 25);
                $search = $request->get('search.value', '');

                // Apply search
                if (!empty($search)) {
                    $query->where(function($q) use ($search) {
                        $q->whereHas('fromProject', function($subQ) use ($search) {
                            $subQ->where('project_name', 'like', "%{$search}%");
                        })
                        ->orWhereHas('toProject', function($subQ) use ($search) {
                            $subQ->where('project_name', 'like', "%{$search}%");
                        })
                        ->orWhereHas('transferredBy', function($subQ) use ($search) {
                            $subQ->where('name', 'like', "%{$search}%");
                        });
                    });
                }

                $totalRecords = MaterialSiteTransfer::count();
                $filteredRecords = $query->count();

                $transfers = $query->offset($start)->limit($length)->get();

                $data = [];
                foreach ($transfers as $transfer) {
                    $colors = [
                        'pending' => 'warning',
                        'completed' => 'success',
                        'cancelled' => 'danger'
                    ];
                    $color = $colors[$transfer->status] ?? 'secondary';

                    $data[] = [
                        'transfer_number' => 'ST-' . str_pad($transfer->id, 5, '0', STR_PAD_LEFT),
                        'transfer_date' => $transfer->transfer_date ? $transfer->transfer_date->format('M d, Y') : 'N/A',
                        'from_project' => $transfer->fromProject ? $transfer->fromProject->project_name : 'N/A',
                        'to_project' => $transfer->toProject ? $transfer->toProject->project_name : 'N/A',
                        'items_count' => '<span class="badge badge-info">' . $transfer->items()->count() . ' items</span>',
                        'total_value' => number_format($transfer->total_value ?: 0, 2),
                        'transferred_by' => $transfer->transferredBy ? $transfer->transferredBy->name : 'N/A',
                        'status' => '<span class="badge badge-' . $color . '">' . ucfirst($transfer->status ?: 'unknown') . '</span>',
                        'actions' => '<div class="d-flex align-items-center" style="white-space: nowrap;">
                                        <a href="' . route('warehouse.site-transfers.show', $transfer) . '" class="btn btn-sm btn-info mr-1" title="View Transfer">
                                            <i class="material-icons mr-1" style="font-size: 14px;">visibility</i>
                                            View
                                        </a>
                                        <button onclick="printSingleTransfer(' . $transfer->id . ')" class="btn btn-sm btn-outline-primary" title="Print Transfer">
                                            <i class="material-icons" style="font-size: 14px;">print</i>
                                        </button>
                                      </div>'
                    ];
                }

                return response()->json([
                    'draw' => intval($request->get('draw')),
                    'recordsTotal' => $totalRecords,
                    'recordsFiltered' => $filteredRecords,
                    'data' => $data
                ]);

            } catch (\Exception $e) {
                return response()->json([
                    'error' => 'Error loading transfers: ' . $e->getMessage()
                ], 500);
            }
        }

        // Calculate stats for the page
        $stats = [
            'total_transfers' => MaterialSiteTransfer::count(),
            'this_month' => MaterialSiteTransfer::whereMonth('transfer_date', now()->month)
                ->whereYear('transfer_date', now()->year)->count(),
            'completed' => MaterialSiteTransfer::where('status', 'completed')->count(),
            'total_value' => MaterialSiteTransfer::where('status', 'completed')->sum('total_value') ?: 0
        ];

        return view('warehouse.site-transfers.index', compact('stats'));
    }

    /**
     * Show the form for creating a new site transfer.
     */
    public function create()
    {
        $projects = Project::all();
        return view('warehouse.site-transfers.create', compact('projects'));
    }

    /**
     * Get project inventory for the selected project.
     */
    public function getProjectInventory(Request $request)
    {
        $projectId = $request->get('project_id');

        \Log::info("getProjectInventory called for project: " . $projectId);

        $inventory = ProjectInventory::with(['item', 'item.unit'])
            ->where('project_id', $projectId)
            ->where('quantity_available', '>', 0)
            ->get()
            ->map(function ($inv) {
                $availableForTransfer = $inv->quantity_available - $inv->allocated_quantity;
                return [
                    'id' => $inv->id,
                    'item_id' => $inv->item_id,
                    'item_name' => $inv->item->item_description ?? 'Unknown Item',
                    'item_code' => $inv->item->item_code ?? 'N/A',
                    'current_stock' => (float) $inv->quantity_available,
                    'unit' => $inv->item->unit ? ($inv->item->unit->unit_symbol ?? $inv->item->unit->unit_name ?? 'N/A') : 'N/A',
                    'unit_price' => (float) ($inv->unit_price ?? 0),
                    'allocated_stock' => (float) $inv->allocated_quantity,
                    'available_for_transfer' => (float) ($availableForTransfer > 0 ? $availableForTransfer : 0)
                ];
            });

        \Log::info("getProjectInventory returning " . $inventory->count() . " items for project " . $projectId, $inventory->toArray());

        return response()->json($inventory);
    }

    /**
     * Get item allocation details.
     */
    public function getItemAllocation(Request $request)
    {
        try {
            $projectId = $request->get('project_id');
            $itemId = $request->get('item_id');

            $projectInventory = ProjectInventory::with(['item', 'item.unit'])
                ->where('project_id', $projectId)
                ->where('item_id', $itemId)
                ->first();

            if (!$projectInventory) {
                return response()->json(['error' => 'Item not found in project'], 404);
            }

        $availableForTransfer = $projectInventory->quantity_available - $projectInventory->allocated_quantity;

        // Get actual allocation sources from stock movements - where the items originally came from
        $allocations = [];

        // Simplified approach: Get delivery sources without return calculations for now
        // This will show all deliveries to the project, and we'll enhance with return tracking later

        // 1. Check Supplier Deliveries
        $supplierDeliveries = DB::table('incoming_operation_items as ioi')
            ->join('incoming_operations as io', 'ioi.incoming_operation_id', '=', 'io.id')
            ->where('io.project_id', $projectId)
            ->where('io.operation_type', 'supplier_delivery')
            ->where('io.delivery_type', 'project_site')
            ->where('io.status', 'completed')
            ->where('ioi.item_id', $itemId)
            ->where('ioi.quantity_delivered', '>', 0)
            ->select(
                'io.operation_number',
                'io.operation_date',
                'ioi.quantity_delivered',
                'ioi.unit_price',
                'ioi.id as source_item_id'
            )
            ->get();

        foreach ($supplierDeliveries as $delivery) {
            // Calculate returns for this specific supplier delivery item
            $returnedQuantity = DB::table('incoming_operation_items as return_items')
                ->join('incoming_operations as return_ops', 'return_items.incoming_operation_id', '=', 'return_ops.id')
                ->where('return_ops.operation_type', 'site_return')
                ->where('return_ops.status', 'completed')
                ->where('return_items.supplier_delivery_item_id', $delivery->source_item_id)
                ->sum('return_items.quantity_delivered');

            // Calculate site transfers out from this specific reference
            $transferredOut = DB::table('material_site_transfer_items as msti')
                ->join('material_site_transfers as mst', 'msti.site_transfer_id', '=', 'mst.id')
                ->where('mst.from_project_id', $projectId)
                ->where('msti.item_id', $itemId)
                ->where('mst.status', 'completed')
                // Add specific reference tracking for supplier deliveries in the future
                ->sum('msti.quantity_transferred');

            $netAvailable = $delivery->quantity_delivered - $returnedQuantity - $transferredOut;

            // Only include if there's available quantity
            if ($netAvailable > 0) {
                $allocations[] = [
                    'id' => 'SD-' . $delivery->source_item_id,
                    'reference' => $delivery->operation_number,
                    'source_type' => 'Supplier Delivery',
                    'quantity_received' => $delivery->quantity_delivered,
                    'quantity_allocated' => $returnedQuantity + $transferredOut,
                    'quantity_available' => $netAvailable,
                    'unit_price' => $delivery->unit_price ?? 0,
                    'delivery_date' => $delivery->operation_date
                ];
            }
        }

        // 2. Check Material Requests delivered to this project
        $materialRequests = DB::table('outgoing_items as oi')
            ->join('outgoing_transactions as ot', 'oi.outgoing_transaction_id', '=', 'ot.id')
            ->where('ot.project_id', $projectId)
            ->whereIn('ot.status', ['successful', 'processed_multi_source'])
            ->where('oi.item_id', $itemId)
            ->where('oi.quantity_released', '>', 0)
            ->select(
                'ot.material_request_number as request_number',
                'ot.transaction_date',
                'oi.quantity_released',
                'oi.unit_price',
                'oi.id as source_item_id',
                'ot.id as transaction_id'
            )
            ->get();

        foreach ($materialRequests as $request) {
            // Calculate returns for this specific material request item
            $returnedQuantity = DB::table('incoming_operation_items as return_items')
                ->join('incoming_operations as return_ops', 'return_items.incoming_operation_id', '=', 'return_ops.id')
                ->where('return_ops.operation_type', 'site_return')
                ->where('return_ops.status', 'completed')
                ->where('return_items.outgoing_item_id', $request->source_item_id)
                ->sum('return_items.quantity_delivered');

            // Calculate site transfers out from this specific reference (for future enhancement)
            $transferredOut = 0; // Placeholder for future reference-specific tracking

            $netAvailable = $request->quantity_released - $returnedQuantity - $transferredOut;

            // Only include if there's available quantity
            if ($netAvailable > 0) {
                $allocations[] = [
                    'id' => 'MR-' . $request->source_item_id,
                    'reference' => ($request->request_number ?? 'MR-' . $request->transaction_id) . ' #' . $request->transaction_id,
                    'source_type' => 'Material Request',
                    'quantity_received' => $request->quantity_released,
                    'quantity_allocated' => $returnedQuantity + $transferredOut,
                    'quantity_available' => $netAvailable,
                    'unit_price' => $request->unit_price ?? 0,
                    'delivery_date' => $request->transaction_date
                ];
            }
        }

        // 3. Check Direct Deliveries
        $directDeliveries = DB::table('direct_delivery_items as ddi')
            ->join('direct_deliveries as dd', 'ddi.direct_delivery_id', '=', 'dd.id')
            ->where('dd.project_id', $projectId)
            ->whereIn('dd.status', ['delivered', 'received', 'partially_received'])
            ->where('ddi.item_id', $itemId)
            ->where('ddi.quantity_delivered', '>', 0)
            ->select(
                'dd.reference_number',
                'dd.delivery_date',
                'ddi.quantity_delivered',
                'ddi.unit_price',
                'ddi.id as source_item_id'
            )
            ->get();

        foreach ($directDeliveries as $delivery) {
            // Calculate returns for this specific direct delivery item
            $returnedQuantity = DB::table('incoming_operation_items as return_items')
                ->join('incoming_operations as return_ops', 'return_items.incoming_operation_id', '=', 'return_ops.id')
                ->where('return_ops.operation_type', 'site_return')
                ->where('return_ops.status', 'completed')
                ->where('return_items.direct_delivery_item_id', $delivery->source_item_id)
                ->sum('return_items.quantity_delivered');

            // Calculate site transfers out from this specific reference (for future enhancement)
            $transferredOut = 0; // Placeholder for future reference-specific tracking

            $netAvailable = $delivery->quantity_delivered - $returnedQuantity - $transferredOut;

            // Only include if there's available quantity
            if ($netAvailable > 0) {
                $allocations[] = [
                    'id' => 'DD-' . $delivery->source_item_id,
                    'reference' => $delivery->reference_number,
                    'source_type' => 'Direct Delivery',
                    'quantity_received' => $delivery->quantity_delivered,
                    'quantity_allocated' => $returnedQuantity + $transferredOut,
                    'quantity_available' => $netAvailable,
                    'unit_price' => $delivery->unit_price ?? 0,
                    'delivery_date' => $delivery->delivery_date
                ];
            }
        }

        // 4. Check Material Transfers TO this project
        $materialTransfers = DB::table('material_site_transfer_items as msti')
            ->join('material_site_transfers as mst', 'msti.site_transfer_id', '=', 'mst.id')
            ->join('projects as from_project', 'mst.from_project_id', '=', 'from_project.id')
            ->where('mst.to_project_id', $projectId)
            ->where('mst.status', 'completed')
            ->where('msti.item_id', $itemId)
            ->where('msti.quantity_transferred', '>', 0)
            ->select(
                'mst.id as transfer_id',
                'mst.transfer_date',
                'msti.quantity_transferred',
                'msti.unit_price',
                'msti.id as source_item_id',
                'from_project.project_name as from_project_name'
            )
            ->get();

        foreach ($materialTransfers as $transfer) {
            // Calculate returns for this specific material transfer item
            $returnedQuantity = DB::table('incoming_operation_items as return_items')
                ->join('incoming_operations as return_ops', 'return_items.incoming_operation_id', '=', 'return_ops.id')
                ->where('return_ops.operation_type', 'site_return')
                ->where('return_ops.status', 'completed')
                ->where('return_items.material_transfer_item_id', $transfer->source_item_id)
                ->sum('return_items.quantity_delivered');

            // Calculate site transfers out from this specific reference (for future enhancement)
            $transferredOut = 0; // Placeholder for future reference-specific tracking

            $netAvailable = $transfer->quantity_transferred - $returnedQuantity - $transferredOut;

            // Only include if there's available quantity
            if ($netAvailable > 0) {
                $allocations[] = [
                    'id' => 'MT-' . $transfer->source_item_id,
                    'reference' => 'MT-' . $transfer->transfer_id,
                    'source_type' => 'Material Transfer',
                    'quantity_received' => $transfer->quantity_transferred,
                    'quantity_allocated' => $returnedQuantity + $transferredOut,
                    'quantity_available' => $netAvailable,
                    'unit_price' => $transfer->unit_price ?? 0,
                    'delivery_date' => $transfer->transfer_date,
                    'from_project' => $transfer->from_project_name
                ];
            }
        }

        // If no specific allocations found, fall back to generic project inventory
        if (empty($allocations) && $availableForTransfer > 0) {
            $allocations[] = [
                'id' => $projectInventory->id,
                'reference' => 'Project Stock - ' . $projectInventory->project->project_name,
                'source_type' => 'Project Inventory',
                'quantity_received' => $projectInventory->quantity_available + $projectInventory->allocated_quantity,
                'quantity_allocated' => $projectInventory->allocated_quantity,
                'quantity_available' => $availableForTransfer,
                'unit_price' => $projectInventory->unit_price ?? 0,
                'delivery_date' => $projectInventory->created_at->format('Y-m-d')
            ];
        }

            return response()->json([
                'item' => [
                    'name' => $projectInventory->item->item_description ?? 'Unknown Item',
                    'code' => $projectInventory->item->item_code ?? 'N/A',
                    'unit' => $projectInventory->item->unit ? ($projectInventory->item->unit->unit_symbol ?? $projectInventory->item->unit->unit_name ?? 'N/A') : 'N/A',
                    'current_stock' => $projectInventory->quantity_available,
                    'allocated_stock' => $projectInventory->allocated_quantity,
                    'available_for_transfer' => $availableForTransfer
                ],
                'allocations' => $allocations
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getItemAllocation: ' . $e->getMessage(), [
                'project_id' => $request->get('project_id'),
                'item_id' => $request->get('item_id'),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => 'Error loading allocation details: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created site transfer.
     */
    public function store(Request $request)
    {
        // Log incoming request for debugging
        \Log::info('Site transfer store request', $request->all());

        // Decode JSON items if sent as string
        if ($request->has('items') && is_string($request->input('items'))) {
            $items = json_decode($request->input('items'), true);
            $request->merge(['items' => $items]);
            \Log::info('Decoded items from JSON', ['items' => $items]);
        }

        try {
            $validated = $request->validate([
            'from_project_id' => 'required|exists:projects,id',
            'to_project_id' => 'required|exists:projects,id|different:from_project_id',
            'transfer_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.project_inventory_id' => 'required|exists:project_inventory,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.allocations' => 'required|array|min:1',
            'items.*.allocations.*.inventory_id' => 'required|string',
            'items.*.allocations.*.quantity' => 'required|numeric|min:0.01'
        ]);

            \Log::info('Validation passed, starting transaction', $validated);

        DB::transaction(function () use ($validated) {
            // Create site transfer
            $transfer = MaterialSiteTransfer::create([
                'from_project_id' => $validated['from_project_id'],
                'to_project_id' => $validated['to_project_id'],
                'transfer_date' => $validated['transfer_date'],
                'notes' => $validated['notes'] ?? null,
                'transferred_by' => Auth::id(),
                'status' => 'completed'
            ]);

            $totalValue = 0;

            // Process each item
            foreach ($validated['items'] as $itemData) {
                $projectInventory = ProjectInventory::find($itemData['project_inventory_id']);
                $transferQuantity = $itemData['quantity'];
                $itemTotalValue = 0;

                // Store original quantities before updates
                $sourceQuantityBefore = $projectInventory->quantity_available;

                // Create transfer item
                $transferItem = MaterialSiteTransferItem::create([
                    'site_transfer_id' => $transfer->id,
                    'item_id' => $projectInventory->item_id,
                    'project_inventory_id' => $itemData['project_inventory_id'],
                    'quantity_transferred' => $transferQuantity,
                    'unit_price' => $projectInventory->unit_price,
                    'total_value' => $transferQuantity * $projectInventory->unit_price
                ]);

                $itemTotalValue += $transferItem->total_value;

                // Update source project inventory
                $projectInventory->quantity_available -= $transferQuantity;
                $projectInventory->save();

                // Create or update destination project inventory
                $destinationInventory = ProjectInventory::firstOrCreate(
                    [
                        'project_id' => $validated['to_project_id'],
                        'item_id' => $projectInventory->item_id
                    ],
                    [
                        'quantity_available' => 0,
                        'allocated_quantity' => 0,
                        'unit_price' => $projectInventory->unit_price
                    ]
                );

                // Store destination quantity before update
                $destinationQuantityBefore = $destinationInventory->quantity_available;
                $destinationInventory->quantity_available += $transferQuantity;
                $destinationInventory->save();

                // Create stock movement for source project (outgoing)
                StockMovement::create([
                    'item_id' => $projectInventory->item_id,
                    'movement_type' => 'transfer_out',
                    'reference_type' => 'site_transfer',
                    'reference_id' => $transfer->id,
                    'quantity_before' => $sourceQuantityBefore,
                    'quantity_moved' => $transferQuantity,
                    'quantity_after' => $projectInventory->quantity_available,
                    'division_id' => Auth::user()->division_id ?? 1,
                    'user_id' => Auth::id(),
                    'notes' => "Site transfer to project: " . \App\Models\Warehouse\Project::find($validated['to_project_id'])->project_name
                ]);

                // Create stock movement for destination project (incoming)
                StockMovement::create([
                    'item_id' => $projectInventory->item_id,
                    'movement_type' => 'transfer_in',
                    'reference_type' => 'site_transfer',
                    'reference_id' => $transfer->id,
                    'quantity_before' => $destinationQuantityBefore,
                    'quantity_moved' => $transferQuantity,
                    'quantity_after' => $destinationInventory->quantity_available,
                    'division_id' => Auth::user()->division_id ?? 1,
                    'user_id' => Auth::id(),
                    'notes' => "Site transfer from project: " . \App\Models\Warehouse\Project::find($validated['from_project_id'])->project_name
                ]);

                $totalValue += $itemTotalValue;
            }

            // Update transfer total
            $transfer->update(['total_value' => $totalValue]);
        });

            \Log::info('Site transfer transaction completed successfully');

        return redirect()->route('warehouse.site-transfers.index')
            ->with('success', 'Site transfer completed successfully.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Site transfer validation failed', ['errors' => $e->errors()]);
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Site transfer failed', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return redirect()->back()
                ->with('error', 'Transfer failed: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified site transfer.
     */
    public function show(MaterialSiteTransfer $siteTransfer)
    {
        $siteTransfer->load([
            'fromProject',
            'toProject',
            'transferredBy',
            'items.item'
        ]);

        return view('warehouse.site-transfers.show', compact('siteTransfer'));
    }

    /**
     * Print site transfers report.
     */
    public function print(Request $request)
    {
        $query = MaterialSiteTransfer::with([
            'fromProject',
            'toProject',
            'transferredBy',
            'items'
        ]);

        // Apply same search logic as the main index method
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transfer_number', 'like', "%{$search}%")
                  ->orWhereHas('fromProject', function ($subQuery) use ($search) {
                      $subQuery->where('project_name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('toProject', function ($subQuery) use ($search) {
                      $subQuery->where('project_name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('transferredBy', function ($subQuery) use ($search) {
                      $subQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Apply filters (same as main index method)
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->where('transfer_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('transfer_date', '<=', $request->date_to);
        }

        // Get ALL transfers (no pagination for print)
        $transfers = $query->orderBy('transfer_date', 'desc')->get();

        return view('warehouse.site-transfers.print', compact('transfers'));
    }

    /**
     * Print single site transfer.
     */
    public function printSingle(MaterialSiteTransfer $siteTransfer)
    {
        $siteTransfer->load([
            'fromProject',
            'toProject',
            'transferredBy',
            'items.item'
        ]);

        return view('warehouse.site-transfers.print-single', compact('siteTransfer'));
    }
}