<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\OutgoingItem;
use App\Models\Warehouse\OutgoingFulfillment;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\Supplier;
use App\Models\Warehouse\Inventory;
use App\Models\DirectDelivery;
use App\Models\DirectDeliveryItem;
use App\Models\Warehouse\MaterialTransferRequest;
use App\Models\Warehouse\MaterialTransferRequestItem;
use App\Models\DamagedItem;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class MaterialIssueController extends Controller
{
    /**
     * Display the material issue decision page
     */
    public function index(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get pending material requests that need issue decisions
        // Only show requests that have at least one item with remaining balance
        $query = OutgoingTransaction::with(['project', 'division', 'requester', 'items.item'])
                                   ->whereNotIn('status', ['successful', 'cancelled', 'rejected', 'completed', 'closed'])
                                   ->whereHas('items', function($q) {
                                       $q->where('quantity_balance', '>', 0);
                                   })
                                   ->latest('created_at');

        // Apply filters
        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('date_from')) {
            $query->where('transaction_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('transaction_date', '<=', $request->date_to);
        }

        $pendingRequests = $query->get();

        // Additional filter to remove requests where ALL items have zero balance
        $pendingRequests = $pendingRequests->filter(function($request) {
            $totalBalance = $request->items->sum('quantity_balance');
            return $totalBalance > 0;
        });

        // Manual pagination of filtered results
        $page = request()->get('page', 1);
        $perPage = 15;
        $pendingRequests = new \Illuminate\Pagination\LengthAwarePaginator(
            $pendingRequests->forPage($page, $perPage),
            $pendingRequests->count(),
            $perPage,
            $page,
            ['path' => request()->url()]
        );

        // Debug: Log what we found
        \Log::info('Material Issue Index - Pending Requests', [
            'count' => $pendingRequests->count(),
            'status_check' => OutgoingTransaction::select('id', 'status', 'material_request_number')->get()->toArray()
        ]);

        // Get filter options
        $projects = Project::all();

        // Get statistics
        $stats = [
            'pending_decisions' => OutgoingTransaction::whereNotIn('status', ['successful', 'cancelled', 'rejected', 'completed', 'closed'])
                                    ->whereHas('items', function($q) {
                                        $q->where('quantity_balance', '>', 0);
                                    })->count(),
            'inventory_issues' => OutgoingTransaction::where('status', 'successful')->count(),
            'direct_deliveries' => \App\Models\DirectDelivery::whereIn('status', ['pending', 'delivered'])->count(),
            'project_transfers' => \App\Models\Warehouse\MaterialTransferRequest::whereIn('status', ['approved', 'partially_approved'])->count()
        ];

        return view('warehouse.material-issue.index', compact(
            'pendingRequests', 'projects', 'stats'
        ));
    }

    /**
     * Show the issue decision form for a specific material request
     */
    public function show(OutgoingTransaction $transaction)
    {
        Gate::authorize('view', $transaction);

        $transaction->load(['project', 'division', 'requester', 'items.item']);

        // Add debug logging for transaction 44
        if ($transaction->id == 44) {
            \Log::info('DEBUG: Loading transaction 44 for material issue', [
                'transaction_id' => $transaction->id,
                'material_request' => $transaction->material_request_number,
                'items_count' => $transaction->items->count()
            ]);
        }

        // Get issue history
        $fulfillments = OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
            ->get()
            ->groupBy('item_id');

        // Get all item IDs for this transaction
        $itemIds = $transaction->items->pluck('item_id')->toArray();

        // FOC: quantity_available IS the available quantity
        $focQuantities = \App\Models\FocInventory::getAvailableStockByItem()
            ->filter(function($qty, $itemId) use ($itemIds) {
                return in_array($itemId, $itemIds);
            });

        // Calculate remaining quantities for each item (ensure all fulfillment types are included)
        foreach ($transaction->items as $item) {
            $issuedQty = 0;
            if (isset($fulfillments[$item->item_id])) {
                $issuedQty = $fulfillments[$item->item_id]->sum('quantity_fulfilled');
            }
            $item->quantity_fulfilled = $issuedQty; // Dynamic property for view
            $item->quantity_remaining = max(0, $item->quantity_requested - $issuedQty);

            // Set FOC available quantity from the bulk query result
            $focAvailable = $focQuantities[$item->item_id] ?? 0;
            $item->foc_available = $focAvailable;

            // Also set it using setAttribute to ensure it's properly set
            $item->setAttribute('foc_available', $focAvailable);

            // Debug logging for transaction 44 / item 2109
            if ($transaction->id == 44 && $item->item_id == 2109) {
                \Log::info('DEBUG: Item 2109 in transaction 44', [
                    'item_id' => $item->item_id,
                    'item_code' => $item->item->item_code ?? 'N/A',
                    'foc_available_set' => $item->foc_available,
                    'quantity_requested' => $item->quantity_requested,
                    'quantity_remaining' => $item->quantity_remaining
                ]);
            }
        }

        // Get options for different issue types
        $projects = Project::where('id', '!=', $transaction->project_id)->get();
        $suppliers = Supplier::where('status', 'active')->get();
        // Use ProjectDivision for divisions since inventory references project_divisions table
        $divisions = \App\Models\Warehouse\ProjectDivision::where('status', 'active')->orderBy('division_name')->get();

        // Use multi-source view for better flexibility
        return view('warehouse.material-issue.decision-multi', compact(
            'transaction', 'projects', 'suppliers', 'divisions'
        ));
    }

    /**
     * Process the issue decision
     */
    public function processDecision(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'issue_type' => 'required|in:inventory_dispatch,direct_delivery,project_transfer,foc_transfer',
            'notes' => 'nullable|string',
        ]);

        $issueType = $request->issue_type;

        switch ($issueType) {
            case 'inventory_dispatch':
                return $this->processInventoryDispatch($request, $transaction);

            case 'direct_delivery':
                return $this->processDirectDelivery($request, $transaction);

            case 'project_transfer':
                return $this->processProjectTransfer($request, $transaction);

            default:
                return back()->with('error', 'Invalid issue type selected.');
        }
    }

    /**
     * Process inventory dispatch decision
     */
    private function processInventoryDispatch(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'dispatch_date' => 'required|date|after_or_equal:today',
            'receiver_name' => 'required|string|max:255',
            'items' => 'required|array',
            'items.*.quantity_to_issue' => 'required|numeric|min:0.01',
        ]);

        // Update transaction items with quantities to issue
        foreach ($request->items as $itemId => $itemData) {
            $outgoingItem = $transaction->items()->where('item_id', $itemId)->first();
            if ($outgoingItem) {
                $outgoingItem->update([
                    'quantity_released' => $itemData['quantity_to_issue']
                ]);
            }
        }

        // Update transaction status and dispatch
        $transaction->update([
            'status' => 'processing',
            'receiver_name' => $request->receiver_name,
            'notes' => $request->notes
        ]);

        // Process the dispatch
        $transaction->approveAndDispatch(auth()->id());

        return redirect()->route('warehouse.material-issue.index')
            ->with('success', 'Material request processed for inventory dispatch successfully.');
    }

    /**
     * Process direct delivery decision
     */
    private function processDirectDelivery(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'supplier_id' => 'nullable|exists:suppliers,id',
            'delivery_date' => 'required|date',
            'lpo_number' => 'nullable|string|max:255',
            'delivery_note_number' => 'nullable|string|max:255',
        ]);

        // Create Direct Delivery based on the material request
        $directDelivery = \App\Models\DirectDelivery::create([
            'delivery_date' => $request->delivery_date,
            'project_id' => $transaction->project_id,
            'project_division_id' => $transaction->division_id,
            'supplier_id' => $request->supplier_id,
            'purchase_order_number' => $request->lpo_number,
            'operation_type' => 'supplier_delivery',
            'delivery_type' => 'purchase_order',
            'workflow_status' => 'pending_supplier',
            'status' => 'pending',
            'material_request_id' => null,
            'currency' => 'AED',
            'exchange_rate' => 1.0000,
            'vat_rate' => 5.00,
        ]);

        // Create corresponding Incoming Transaction for dual tracking
        $incomingTransaction = \App\Models\Warehouse\IncomingTransaction::create([
            'transaction_date' => $request->delivery_date,
            'transaction_type' => 'direct_delivery',
            'supplier_id' => $request->supplier_id,
            'project_id' => $transaction->project_id,
            'project_number' => $transaction->project->project_number ?? null,
            'type_of_goods' => 'Material Request - Direct Delivery',
            'lpo_number' => $request->lpo_number,
            'delivery_note_number' => $request->delivery_note_number,
            'division_id' => \App\Models\Department::first()->id ?? 1, // Use first available department ID
            'total_amount' => 0, // Will be calculated from items
            'received_by' => auth()->id(),
            'received_by_name' => auth()->user()->name,
            'status' => 'pending',
            'notes' => "Direct delivery for material request: {$transaction->material_request_number}. " . $request->notes
        ]);

        $totalAmount = 0;

        // Create direct delivery items and corresponding incoming items
        foreach ($transaction->items as $outgoingItem) {
            $itemTotal = $outgoingItem->quantity_requested * ($outgoingItem->unit_price ?: 0);
            $totalAmount += $itemTotal;

            // Create direct delivery item
            \App\Models\DirectDeliveryItem::create([
                'direct_delivery_id' => $directDelivery->id,
                'item_id' => $outgoingItem->item_id,
                'quantity_ordered' => $outgoingItem->quantity_requested,
                'ordered_quantity' => $outgoingItem->quantity_requested,
                'unit_price' => $outgoingItem->unit_price ?: 0,
                'unit_price_aed' => $outgoingItem->unit_price ?: 0,
                'currency' => 'AED',
                'status' => 'pending',
            ]);

            // Create corresponding incoming item for dual tracking
            \App\Models\Warehouse\IncomingItem::create([
                'incoming_transaction_id' => $incomingTransaction->id,
                'item_id' => $outgoingItem->item_id,
                'quantity_received' => $outgoingItem->quantity_requested,
                'unit_price' => $outgoingItem->unit_price ?: 0,
                'total_price' => $itemTotal,
                'stock_status' => 'pending_delivery'
            ]);
        }

        // Update incoming transaction total amount
        $incomingTransaction->update(['total_amount' => $totalAmount]);

        // Update original transaction status
        $transaction->update([
            'status' => 'converted_to_direct_delivery',
            'notes' => $request->notes
        ]);

        return redirect()->route('warehouse.direct-deliveries.show', $directDelivery->id)
            ->with('success', 'Material request converted to direct delivery. Please update supplier and delivery details.');
    }

    /**
     * Process project transfer decision
     */
    private function processProjectTransfer(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'source_project_id' => 'required|exists:projects,id|different:' . $transaction->project_id,
            'transfer_reason' => 'required|string|max:255',
            'required_date' => 'required|date|after:today',
            'priority' => 'required|in:low,medium,high,urgent',
        ]);

        // Create Material Transfer Request
        $transferRequest = \App\Models\Warehouse\MaterialTransferRequest::create([
            'request_date' => now(),
            'transferrer_project_id' => $request->source_project_id,
            'receiver_project_id' => $transaction->project_id,
            'transfer_reason' => $request->transfer_reason,
            'transfer_notes' => $request->notes,
            'priority' => $request->priority,
            'required_date' => $request->required_date,
            'requested_by' => auth()->id(),
            'status' => 'pending_transferrer_approval'
        ]);

        // Create transfer request items
        foreach ($transaction->items as $outgoingItem) {
            // Check available quantity in source project
            $projectInventory = \App\Models\Warehouse\ProjectInventory::where('project_id', $request->source_project_id)
                ->where('item_id', $outgoingItem->item_id)
                ->first();

            $availableQuantity = $projectInventory ? $projectInventory->available_for_transfer : 0;
            $estimatedUnitPrice = $projectInventory ? $projectInventory->unit_price : ($outgoingItem->unit_price ?: 0);

            \App\Models\Warehouse\MaterialTransferRequestItem::create([
                'transfer_request_id' => $transferRequest->id,
                'item_id' => $outgoingItem->item_id,
                'quantity_requested' => $outgoingItem->quantity_requested,
                'quantity_available' => $availableQuantity,
                'estimated_unit_price' => $estimatedUnitPrice,
                'transfer_notes' => "Generated from material request: {$transaction->material_request_number}"
            ]);
        }

        // Update original transaction status
        $transaction->update([
            'status' => 'converted_to_transfer',
            'notes' => $request->notes
        ]);

        return redirect()->route('warehouse.transfer-requests.show', $transferRequest->id)
            ->with('success', 'Material request converted to project transfer. Awaiting source project approval.');
    }

    /**
     * Process multi-source material issue decision
     */
    public function processMultiSource(Request $request, OutgoingTransaction $transaction)
    {
        // Debug: Check if method is being called
        if ($request->has('debug')) {
            return response()->json([
                'message' => 'Method reached successfully',
                'transaction_id' => $transaction->id,
                'request_data' => $request->all()
            ]);
        }

        \Log::info('processMultiSource called', [
            'transaction_id' => $transaction->id,
            'request_data' => $request->all()
        ]);

        // Build validation rules dynamically based on enabled sections
        $validationRules = [
            'enable_inventory' => 'required|in:0,1',
            'enable_delivery' => 'required|in:0,1',
            'enable_foc' => 'required|in:0,1',
            'inventory' => 'nullable|array',
            'delivery' => 'nullable|array',
            'foc' => 'nullable|array',
            'inventory.*' => 'nullable|numeric|min:0',
            'delivery.*' => 'nullable|numeric|min:0',
            'foc.*' => 'nullable|numeric|min:0',
            'inventory_division' => 'nullable|array',
            'inventory_division.*' => 'nullable|exists:departments,id',
            'delivery_division' => 'nullable|array',
            'delivery_division.*' => 'nullable|exists:departments,id',
            'inventory_file' => 'nullable|file|max:10240|mimes:pdf,doc,docx,jpg,jpeg,png',
            'delivery_file' => 'nullable|file|max:10240|mimes:pdf,doc,docx,jpg,jpeg,png',
            'delivery_supplier_id' => 'nullable|required_if:enable_delivery,1|exists:suppliers,id',
            // FOC doesn't require reference selection - it works like inventory dispatch
            'foc_transfer_date' => 'required_if:enable_foc,1|date',
        ];

        // Add unit price and division validation when direct delivery is enabled
        if ($request->enable_delivery == '1') {
            // Get delivery items to validate unit prices and divisions for items with quantities
            $deliveryItems = $request->input('delivery', []);
            foreach ($deliveryItems as $itemId => $quantity) {
                if ($quantity > 0) {
                    $validationRules["delivery_unit_price.{$itemId}"] = 'required|numeric|min:0';
                    $validationRules["delivery_division.{$itemId}"] = 'required|exists:departments,id';
                }
            }
        }

        $request->validate($validationRules);

        \Log::info('Validation passed');

        // Handle file uploads
        $inventoryFile = null;
        $deliveryFile = null;

        if ($request->hasFile('inventory_file')) {
            $file = $request->file('inventory_file');
            $filename = 'inventory_' . time() . '_' . $file->getClientOriginalName();
            $uploadPath = 'uploads/material-issue/inventory';

            // Create directory if it doesn't exist
            $fullPath = public_path($uploadPath);
            if (!file_exists($fullPath)) {
                mkdir($fullPath, 0755, true);
            }

            // Move file to public directory
            $file->move($fullPath, $filename);
            $inventoryFile = $uploadPath . '/' . $filename;
        }

        if ($request->hasFile('delivery_file')) {
            $file = $request->file('delivery_file');
            $filename = 'delivery_' . time() . '_' . $file->getClientOriginalName();
            $uploadPath = 'uploads/material-issue/delivery';

            // Create directory if it doesn't exist
            $fullPath = public_path($uploadPath);
            if (!file_exists($fullPath)) {
                mkdir($fullPath, 0755, true);
            }

            // Move file to public directory
            $file->move($fullPath, $filename);
            $deliveryFile = $uploadPath . '/' . $filename;
        }


        // Convert string values to boolean
        $enableInventory = (bool) $request->input('enable_inventory');
        $enableDelivery = (bool) $request->input('enable_delivery');
        $enableFoc = (bool) $request->input('enable_foc');

        // Ensure at least one issue method is enabled
        if (!$enableInventory && !$enableDelivery && !$enableFoc) {
            return back()->with('error', 'Please enable at least one issue method.');
        }

        \Log::info('Multi-source form data:', [
            'enable_inventory' => $request->input('enable_inventory'),
            'enable_delivery' => $request->input('enable_delivery'),
            'enable_foc' => $request->input('enable_foc'),
            'converted_booleans' => [$enableInventory, $enableDelivery, $enableFoc],
            'inventory_data' => $request->inventory,
            'delivery_data' => $request->delivery,
            'foc_data' => $request->foc
        ]);

        try {
            \DB::beginTransaction();

            $results = [];
            $inventoryTotal = 0;
            $deliveryTotal = 0;

            // Initialize arrays to track what's being fulfilled
            $inventoryItems = [];
            $deliveryItems = [];
            $transferItems = [];

        // Process Inventory Dispatch if enabled
        if ($enableInventory && !empty($request->inventory)) {
            $inventoryItems = array_filter($request->inventory, function($qty) { return $qty > 0; });
            if (!empty($inventoryItems)) {
                $inventoryDivisions = $request->inventory_division ?: [];
                $inventoryResult = $this->processInventoryPortion($request, $transaction, $inventoryItems, $inventoryDivisions);
                $results[] = $inventoryResult;
                $inventoryTotal = array_sum($inventoryItems);
            }
        }

        // Process Direct Delivery if enabled
        if ($enableDelivery && !empty($request->delivery)) {
            $deliveryItems = array_filter($request->delivery, function($qty) { return $qty > 0; });
            if (!empty($deliveryItems)) {
                $deliveryResult = $this->processDirectDeliveryPortion($request, $transaction, $deliveryItems);
                $results[] = $deliveryResult;
                $deliveryTotal = array_sum($deliveryItems);
            }
        }


        // Process FOC Transfer if enabled
        if ($enableFoc && !empty($request->foc)) {
            $focItems = array_filter($request->foc, function($qty) { return $qty > 0; });
            if (!empty($focItems)) {
                $focResult = $this->processFocTransferPortion($request, $transaction, $focItems);
                $results[] = $focResult;
                $focTotal = array_sum($focItems);
            }
        }

        // Check if all items are fully fulfilled
        $fullyFulfilled = true;
        foreach ($transaction->items as $item) {
            $totalFulfilled = 0;

            // Calculate total fulfilled for this item across all methods
            if (isset($inventoryItems[$item->item_id])) {
                $totalFulfilled += $inventoryItems[$item->item_id];
            }
            if (isset($deliveryItems[$item->item_id])) {
                $totalFulfilled += $deliveryItems[$item->item_id];
            }
            if (isset($transferItems[$item->item_id])) {
                $totalFulfilled += $transferItems[$item->item_id];
            }

            // Check if this item is fully fulfilled
            if ($totalFulfilled < $item->quantity_requested) {
                $fullyFulfilled = false;
                break;
            }
        }

        // Update original transaction status based on issue progress
        if ($fullyFulfilled) {
            $transaction->update([
                'status' => 'processed_multi_source',
                'notes' => "Fully fulfilled via multi-source: Inventory({$inventoryTotal}), Delivery({$deliveryTotal})"
            ]);
        } else {
            // Keep as pending if partially fulfilled
            $existingNotes = $transaction->notes ?? '';
            $newNote = "Partially issued: Inventory({$inventoryTotal}), Delivery({$deliveryTotal})";
            $transaction->update([
                'status' => 'pending',
                'notes' => $existingNotes . "\n" . date('Y-m-d H:i') . " - " . $newNote
            ]);
        }

        // Update project inventory for ALL fulfilled items
        $this->updateProjectInventoryForMaterialIssue($transaction, $inventoryItems, $deliveryItems, []);

        // Store file records
        if ($inventoryFile && $enableInventory) {
            \App\Models\Warehouse\MaterialIssueFile::create([
                'outgoing_transaction_id' => $transaction->id,
                'issue_type' => 'inventory',
                'file_path' => $inventoryFile,
                'original_filename' => $request->file('inventory_file')->getClientOriginalName(),
                'uploaded_by' => auth()->id()
            ]);
        }

        if ($deliveryFile && $enableDelivery) {
            \App\Models\Warehouse\MaterialIssueFile::create([
                'outgoing_transaction_id' => $transaction->id,
                'issue_type' => 'delivery',
                'file_path' => $deliveryFile,
                'original_filename' => $request->file('delivery_file')->getClientOriginalName(),
                'uploaded_by' => auth()->id()
            ]);
        }


            // Create summary message
            $messages = [];
            if ($inventoryTotal > 0) $messages[] = "Inventory Dispatch: {$inventoryTotal} units processed";
            if ($deliveryTotal > 0) $messages[] = "Direct Delivery: {$deliveryTotal} units ordered";

            \DB::commit();

            \Log::info('Multi-source material issue completed successfully', [
                'transaction_id' => $transaction->id,
                'inventory_total' => $inventoryTotal,
                'delivery_total' => $deliveryTotal
            ]);

            return redirect()->route('warehouse.material-issue.index')
                ->with('success', 'Multi-source material issue processed successfully! ' . implode(', ', $messages));

        } catch (\Exception $e) {
            \DB::rollback();
            \Log::error('Multi-source material issue failed', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()
                ->with('error', 'Failed to process multi-source material issue: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Process inventory portion of multi-source material issue
     */
    private function processInventoryPortion(Request $request, OutgoingTransaction $originalTransaction, array $inventoryItems, array $inventoryDivisions = [])
    {
        // Create a new outgoing transaction for inventory portion
        $inventoryTransaction = OutgoingTransaction::create([
            'transaction_date' => $request->inventory_dispatch_date ?: now(),
            'project_id' => $originalTransaction->project_id,
            'division_id' => $originalTransaction->division_id,
            'material_request_number' => $originalTransaction->material_request_number . '-INV',
            'type_of_goods' => 'Material Request - Inventory',
            'requested_by' => $originalTransaction->requested_by,
            'receiver_name' => $request->inventory_receiver_name,
            'status' => 'processing',
            'notes' => $request->inventory_notes ?: 'Multi-source: Inventory portion'
        ]);

        // Create inventory transaction items
        foreach ($inventoryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                $originalItem = $originalTransaction->items()->where('item_id', $itemId)->first();
                $divisionId = $inventoryDivisions[$itemId] ?? null;

                // Get average price from the specific division if available
                $averagePrice = Inventory::where('item_id', $itemId)
                    ->when($divisionId, function($query, $divisionId) {
                        return $query->where('division_id', $divisionId);
                    })
                    ->where('quantity_available', '>', 0)
                    ->avg('unit_price') ?: 0;

                OutgoingItem::create([
                    'outgoing_transaction_id' => $inventoryTransaction->id,
                    'item_id' => $itemId,
                    'division_id' => $divisionId,
                    'quantity_requested' => $quantity,
                    'quantity_released' => $quantity,
                    'unit_price' => $averagePrice,
                ]);
            }
        }

        // Process the dispatch immediately
        $inventoryTransaction->approveAndDispatch(auth()->id());

        // Record issue records
        foreach ($inventoryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'inventory',
                    'reference_id' => $inventoryTransaction->id,
                    'reference_type' => 'OutgoingTransaction',
                    'notes' => 'Inventory dispatch'
                ]);
            }
        }

        return ['type' => 'inventory', 'id' => $inventoryTransaction->id, 'total' => array_sum($inventoryItems)];
    }

    /**
     * Process direct delivery portion of multi-source material issue
     */
    private function processDirectDeliveryPortion(Request $request, OutgoingTransaction $originalTransaction, array $deliveryItems)
    {
        // Validate that supplier is selected for delivery
        if (empty($request->delivery_supplier_id)) {
            throw new \InvalidArgumentException('Supplier must be selected for direct delivery.');
        }

        // Get the first division from delivery items (use the division selected in the form)
        $selectedDivisionId = null;
        foreach ($deliveryItems as $itemId => $quantity) {
            if ($quantity > 0 && $request->input("delivery_division.{$itemId}")) {
                $selectedDivisionId = $request->input("delivery_division.{$itemId}");
                break;
            }
        }

        // Create Direct Delivery (marked as delivered immediately since supplier delivers directly to site)
        $directDelivery = DirectDelivery::create([
            'delivery_date' => now(), // Delivered on the same day as form submission
            'project_id' => $originalTransaction->project_id,
            'project_division_id' => $selectedDivisionId ?: $originalTransaction->division_id,
            'supplier_id' => $request->delivery_supplier_id,
            'purchase_order_number' => $request->delivery_lpo_number,
            'operation_type' => 'supplier_delivery',
            'delivery_type' => 'purchase_order',
            'workflow_status' => 'completed', // Completed immediately
            'status' => 'delivered', // Delivered status since it goes directly to project site
            'material_request_id' => null,
            'material_request_number' => $originalTransaction->material_request_number,
            'outgoing_transaction_id' => $originalTransaction->id,
            'currency' => 'AED',
            'exchange_rate' => 1.0000,
            'vat_rate' => 5.00,
            'delivered_at' => now(), // Mark as delivered at current time
            'received_by' => auth()->id(), // Current user is marking it as delivered
        ]);

        // Create corresponding Incoming Transaction for proper dual tracking
        $incomingTransaction = \App\Models\Warehouse\IncomingTransaction::create([
            'transaction_date' => $request->delivery_date ?: now()->addDays(7),
            'transaction_type' => 'direct_delivery',
            'supplier_id' => $request->delivery_supplier_id,
            'project_id' => $originalTransaction->project_id,
            'project_number' => $originalTransaction->project->project_number ?? null,
            'type_of_goods' => 'Material Request - Direct Delivery',
            'lpo_number' => $request->delivery_lpo_number,
            'delivery_note_number' => $request->delivery_note_number,
            'division_id' => $selectedDivisionId ?: (\App\Models\Department::first()->id ?? 1),
            'total_amount' => 0, // Will be calculated from items
            'received_by' => auth()->id(),
            'received_by_name' => auth()->user()->name,
            'status' => 'pending',
            'notes' => "Direct delivery for material request: {$originalTransaction->material_request_number}"
        ]);

        $totalAmount = 0;

        // Create direct delivery items and corresponding incoming items
        foreach ($deliveryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                $unitPrice = $request->input("delivery_unit_price.{$itemId}", 0);
                $batchNumber = $request->input("delivery_batch.{$itemId}");
                $productionDate = $request->input("delivery_production_date.{$itemId}");
                $expiryDate = $request->input("delivery_expiry_date.{$itemId}");
                $divisionId = $request->input("delivery_division.{$itemId}");

                // Create direct delivery item (marked as delivered immediately)
                DirectDeliveryItem::create([
                    'direct_delivery_id' => $directDelivery->id,
                    'item_id' => $itemId,
                    'quantity_ordered' => $quantity,
                    'ordered_quantity' => $quantity,
                    'delivered_quantity' => $quantity, // Same as ordered since it's delivered immediately
                    'unit_price' => $unitPrice,
                    'currency' => 'AED',
                    'status' => 'delivered', // Delivered status
                    'batch_number' => $batchNumber,
                    'production_date' => $productionDate ?: null,
                    'expiry_date' => $expiryDate ?: null,
                ]);

                // Create corresponding incoming item for dual tracking
                \App\Models\Warehouse\IncomingItem::create([
                    'incoming_transaction_id' => $incomingTransaction->id,
                    'item_id' => $itemId,
                    'quantity_received' => $quantity,
                    'unit_price' => $unitPrice,
                    'total_price' => $quantity * $unitPrice,
                    'production_date' => $productionDate ?: null,
                    'expiry_date' => $expiryDate ?: null,
                    'batch_number' => $batchNumber,
                    'stock_status' => 'in_stock'
                ]);

                // Create or update project inventory (direct delivery goes straight to project site)
                $existingProjectInventory = \App\Models\Warehouse\ProjectInventory::where('project_id', $originalTransaction->project_id)
                    ->where('item_id', $itemId)
                    ->first();

                if ($existingProjectInventory) {
                    // Update existing project inventory
                    $newTotalQty = $existingProjectInventory->quantity_available + $quantity;
                    $newTotalValue = $existingProjectInventory->total_value + ($quantity * $unitPrice);
                    $newAvgPrice = $newTotalQty > 0 ? $newTotalValue / $newTotalQty : $unitPrice;

                    $existingProjectInventory->quantity_available += $quantity;
                    $existingProjectInventory->unit_price = $newAvgPrice;
                    $existingProjectInventory->total_value = $newTotalValue;
                    $existingProjectInventory->save();
                } else {
                    // Create new project inventory record
                    \App\Models\Warehouse\ProjectInventory::create([
                        'project_id' => $originalTransaction->project_id,
                        'item_id' => $itemId,
                        'quantity_available' => $quantity,
                        'allocated_quantity' => 0,
                        'unit_price' => $unitPrice,
                        'total_value' => $quantity * $unitPrice
                    ]);
                }

                // DO NOT create warehouse inventory record - items went directly to project site
                // This prevents the items from showing as available in warehouse inventory

                $totalAmount += $quantity * $unitPrice;

                // Record division delivery tracking for direct deliveries
                // Note: Direct deliveries go straight to project, NOT to warehouse inventory
                if ($divisionId) {
                    $deliveryDate = $request->delivery_date ?: now()->addDays(7);

                    $record = \App\Models\Warehouse\DivisionDelivery::firstOrNew([
                        'division_id' => $divisionId,
                        'item_id' => $itemId
                    ]);

                    if ($record->exists) {
                        // Update existing record
                        $record->total_quantity_delivered += $quantity;
                        // DO NOT increment current_available_quantity - items went directly to project
                        // Instead, immediately mark them as issued
                        $record->total_quantity_issued += $quantity;
                        $record->total_value += ($quantity * $unitPrice);
                        $record->average_unit_price = $record->total_value / $record->total_quantity_delivered;
                        $record->last_delivery_date = $deliveryDate;
                    } else {
                        // Create new record
                        $record->total_quantity_delivered = $quantity;
                        $record->current_available_quantity = 0; // Not available - went to project
                        $record->total_quantity_issued = $quantity; // Already issued to project
                        $record->average_unit_price = $unitPrice;
                        $record->total_value = $quantity * $unitPrice;
                        $record->first_delivery_date = $deliveryDate;
                        $record->last_delivery_date = $deliveryDate;
                    }

                    $record->last_updated = now();
                    $record->save();
                }

                // Create issue record for direct delivery
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'direct_delivery',
                    'reference_id' => $directDelivery->id,
                    'reference_type' => 'DirectDelivery',
                    'notes' => 'Direct delivery issue'
                ]);
            }
        }

        // Update incoming transaction total amount
        $incomingTransaction->update(['total_amount' => $totalAmount]);

        return ['type' => 'delivery', 'id' => $directDelivery->id, 'total' => array_sum($deliveryItems)];
    }

    /**
     * Process project transfer portion of multi-source material issue
     */
    private function processProjectTransferPortion(Request $request, OutgoingTransaction $originalTransaction, array $transferItems)
    {
        // Parse the selected reference to get type and ID
        $referenceData = $this->parseTransferReference($request->transfer_reference_id ?? null);

        // Create Material Transfer Request
        $transferRequest = MaterialTransferRequest::create([
            'request_date' => now(),
            'transferrer_project_id' => $request->transfer_source_project_id,
            'receiver_project_id' => $originalTransaction->project_id,
            'transfer_reason' => $request->transfer_reason ?: 'Multi-source material issue',
            'transfer_notes' => $request->transfer_notes ?: "Part of multi-source material request. Reference: " . ($request->transfer_reference_id ?: 'N/A'),
            'priority' => $request->transfer_priority ?: 'medium',
            'required_date' => $request->transfer_date ?: now()->addDays(3),
            'requested_by' => auth()->id(),
            'status' => 'approved' // Auto-approve for immediate transfer since quantities are verified
        ]);

        // Create transfer request items
        foreach ($transferItems as $itemId => $quantity) {
            if ($quantity > 0) {
                // Check available quantity in source project
                $projectInventory = ProjectInventory::where('project_id', $request->transfer_source_project_id)
                    ->where('item_id', $itemId)
                    ->first();

                $availableQuantity = $projectInventory ? $projectInventory->available_for_transfer : 0;
                $estimatedUnitPrice = $projectInventory ? $projectInventory->unit_price : 0;

                // Link to the original reference based on type
                $linkingFields = $this->getLinkingFieldsForReference($referenceData, $itemId);

                MaterialTransferRequestItem::create([
                    'transfer_request_id' => $transferRequest->id,
                    'item_id' => $itemId,
                    'quantity_requested' => $quantity,
                    'quantity_approved' => $quantity, // Auto-approve the full quantity
                    'quantity_available' => $availableQuantity,
                    'estimated_unit_price' => $estimatedUnitPrice,
                    'transfer_notes' => "Multi-source material issue portion",
                    'approval_status' => 'approved'
                ] + $linkingFields);

                // CRITICAL: Decrease source project inventory immediately
                $this->decreaseSourceProjectInventory($request->transfer_source_project_id, $itemId, $quantity);

                // Create issue record for project transfer
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'project_transfer',
                    'reference_id' => $transferRequest->id,
                    'reference_type' => 'MaterialTransferRequest',
                    'notes' => 'Project transfer issue'
                ]);
            }
        }

        return ['type' => 'transfer', 'id' => $transferRequest->id, 'total' => array_sum($transferItems)];
    }

    /**
     * Parse transfer reference ID to get type and ID
     */
    private function parseTransferReference($referenceId)
    {
        if (!$referenceId) {
            return ['type' => null, 'id' => null];
        }

        $parts = explode('_', $referenceId);
        if (count($parts) == 2) {
            return ['type' => $parts[0], 'id' => $parts[1]];
        }

        return ['type' => null, 'id' => null];
    }

    /**
     * Get linking fields for reference based on type
     */
    private function getLinkingFieldsForReference($referenceData, $itemId)
    {
        $fields = [];

        if (!$referenceData['type'] || !$referenceData['id']) {
            return $fields;
        }

        switch ($referenceData['type']) {
            case 'MR':
                // Link to Material Request (Outgoing Transaction)
                $outgoingItem = \App\Models\Warehouse\OutgoingItem::where('outgoing_transaction_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($outgoingItem) {
                    $fields['outgoing_item_id'] = $outgoingItem->id;
                }
                break;

            case 'DD':
                // Link to Direct Delivery
                $deliveryItem = \App\Models\DirectDeliveryItem::where('direct_delivery_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($deliveryItem) {
                    $fields['direct_delivery_item_id'] = $deliveryItem->id;
                }
                break;

            case 'MT':
                // Link to Material Transfer
                $transferItem = \App\Models\Warehouse\MaterialTransferRequestItem::where('transfer_request_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($transferItem) {
                    $fields['material_transfer_item_id'] = $transferItem->id;
                }
                break;
        }

        return $fields;
    }

    /**
     * Decrease source project inventory
     */
    private function decreaseSourceProjectInventory($sourceProjectId, $itemId, $quantity)
    {
        $projectInventory = \App\Models\Warehouse\ProjectInventory::where('project_id', $sourceProjectId)
            ->where('item_id', $itemId)
            ->first();

        if ($projectInventory) {
            // Decrease available quantity
            $projectInventory->updateQuantity($quantity, 'subtract');

            \Log::info("Decreased source project inventory", [
                'project_id' => $sourceProjectId,
                'item_id' => $itemId,
                'quantity_decreased' => $quantity,
                'new_available' => $projectInventory->quantity_available
            ]);
        } else {
            \Log::warning("No project inventory found to decrease", [
                'project_id' => $sourceProjectId,
                'item_id' => $itemId,
                'quantity' => $quantity
            ]);
        }
    }

    /**
     * Update project inventory for all issued items across all issue methods
     */
    private function updateProjectInventoryForMaterialIssue(OutgoingTransaction $transaction, array $inventoryItems, array $deliveryItems, array $transferItems)
    {
        if (!$transaction->project_id) {
            return; // No project to update inventory for
        }

        // Combine all issued quantities by item
        $allIssuedItems = [];

        // Add inventory dispatch quantities
        foreach ($inventoryItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Add direct delivery quantities
        foreach ($deliveryItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Add project transfer quantities (these go to project inventory when transfer is completed)
        foreach ($transferItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Update project inventory for each issued item
        foreach ($allIssuedItems as $itemId => $totalIssued) {
            if ($totalIssued > 0) {
                $projectInventory = ProjectInventory::where('project_id', $transaction->project_id)
                    ->where('item_id', $itemId)
                    ->first();

                if ($projectInventory) {
                    // Update existing record
                    $projectInventory->updateQuantity($totalIssued, 'add');
                } else {
                    // Create new record - get unit price from original request
                    $originalItem = $transaction->items()->where('item_id', $itemId)->first();
                    $unitPrice = $originalItem->unit_price ?? 0;

                    ProjectInventory::create([
                        'project_id' => $transaction->project_id,
                        'item_id' => $itemId,
                        'quantity_available' => $totalIssued,
                        'allocated_quantity' => 0,
                        'unit_price' => $unitPrice,
                        'total_value' => $totalIssued * $unitPrice
                    ]);
                }
            }
        }
    }

    /**
     * Get FOC items for specific item IDs
     */
    public function getFocItems(Request $request)
    {
        try {
            $itemIds = $request->input('item_ids', []);

            if (empty($itemIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No item IDs provided'
                ]);
            }

            // Get approved FOC items (damaged items with approved status)
            $focItems = DamagedItem::with(['item'])
                ->where('status', 'approved')
                ->whereIn('item_id', $itemIds)
                ->where('quantity', '>', 0)
                ->get()
                ->map(function($focItem) {
                    return [
                        'id' => $focItem->id,
                        'item_id' => $focItem->item_id,
                        'item_code' => $focItem->item->item_code ?? 'N/A',
                        'item_description' => $focItem->item->item_description ?? 'N/A',
                        'available_quantity' => $focItem->quantity,
                        'damage_description' => $focItem->damage_description,
                        'damage_severity' => $focItem->damage_severity,
                        'source_reference' => $focItem->source_reference ?? 'FOC-' . str_pad($focItem->id, 6, '0', STR_PAD_LEFT),
                        'unit_cost' => 0, // FOC items have no cost
                        'total_cost' => 0
                    ];
                })
                ->groupBy('item_id')
                ->map(function($group) {
                    // Calculate total available quantity per item
                    $totalQuantity = $group->sum('available_quantity');
                    $references = $group->pluck('source_reference')->unique()->implode(', ');

                    return [
                        'item_id' => $group->first()['item_id'],
                        'item_code' => $group->first()['item_code'],
                        'item_description' => $group->first()['item_description'],
                        'total_available' => $totalQuantity,
                        'references' => $references,
                        'foc_items' => $group->toArray()
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $focItems->values()
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching FOC items: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error fetching FOC items: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get available FOC references for dropdown (DEPRECATED - FOC should bypass reference selection)
     */
    public function getFocReferences(Request $request)
    {
        try {
            // FOC items should be treated like inventory dispatch - no reference selection needed
            // Return empty array to indicate no references required
            return response()->json([
                'success' => true,
                'references' => [],
                'message' => 'FOC items do not require reference selection'
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching FOC references: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error fetching FOC references'
            ]);
        }
    }

    /**
     * Get available FOC quantities for items (like inventory dispatch)
     */
    public function getFocAvailableQuantities(Request $request)
    {
        try {
            $itemIds = $request->input('item_ids', []);

            if (empty($itemIds)) {
                return response()->json([
                    'success' => true,
                    'data' => []
                ]);
            }

            // FOC: quantity_available IS the available quantity
            $focQuantities = \App\Models\FocInventory::whereIn('item_id', $itemIds)
                ->selectRaw('item_id, SUM(quantity_available) as total_quantity')
                ->groupBy('item_id')
                ->get()
                ->map(function($focItem) {
                    return [
                        'item_id' => $focItem->item_id,
                        'total_available' => $focItem->total_quantity,
                        'source_type' => 'FOC Inventory'
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $focQuantities
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching FOC available quantities: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error fetching FOC quantities'
            ]);
        }
    }

    /**
     * Process FOC transfer portion of multi-source fulfillment
     */
    private function processFocTransferPortion(Request $request, OutgoingTransaction $transaction, array $focItems)
    {
        try {
            $transferDate = $request->input('foc_transfer_date', now());
            $transferReason = $request->input('foc_transfer_reason', 'FOC material transfer for project requirement');
            $transferNotes = $request->input('foc_notes', '');

            $processedItems = [];

            foreach ($focItems as $itemId => $quantityToTransfer) {
                if ($quantityToTransfer <= 0) continue;

                // Get total available FOC quantity from FOC inventory
                $totalAvailable = \App\Models\FocInventory::getTotalAvailableForItem($itemId);

                if ($totalAvailable < $quantityToTransfer) {
                    throw new \Exception("Insufficient FOC quantity for item ID {$itemId}. Available: {$totalAvailable}, Requested: {$quantityToTransfer}");
                }

                // PERMANENT FIX: Get FOC inventory items using the available() scope
                // This now includes ALL FOC with quantity > 0 except disposed
                $focInventoryItems = \App\Models\FocInventory::forItem($itemId)
                    ->available()
                    ->orderBy('received_at')
                    ->orderBy('created_at')
                    ->get();

                $remainingToTransfer = $quantityToTransfer;

                foreach ($focInventoryItems as $focInventoryItem) {
                    if ($remainingToTransfer <= 0) break;

                    $qtyFromThisItem = min($remainingToTransfer, $focInventoryItem->quantity_available);
                    $remainingToTransfer -= $qtyFromThisItem;

                    // Create fulfillment record for this portion
                    $fulfillment = OutgoingFulfillment::create([
                        'outgoing_transaction_id' => $transaction->id,
                        'item_id' => $itemId,
                        'fulfillment_type' => 'foc_transfer',
                        'quantity_fulfilled' => $qtyFromThisItem,
                        'unit_cost' => 0, // FOC items have no cost
                        'total_cost' => 0,
                        'supplier_id' => null,
                        'source_reference' => $focInventoryItem->source_reference ?? 'FOC-' . str_pad($focInventoryItem->id, 6, '0', STR_PAD_LEFT),
                        'notes' => $transferNotes,
                        'fulfillment_date' => $transferDate,
                        'created_by' => auth()->id(),
                        'metadata' => json_encode([
                            'foc_inventory_id' => $focInventoryItem->id,
                            'condition' => $focInventoryItem->condition,
                            'damage_severity' => $focInventoryItem->damage_severity,
                            'transfer_reason' => $transferReason,
                            'source_damaged_item_id' => $focInventoryItem->source_damaged_item_id
                        ])
                    ]);

                    // Dispatch FOC inventory using the new method with proper tracking
                    $focInventoryItem->dispatch(
                        $qtyFromThisItem,
                        "FOC dispatch for material issue - " . $transaction->material_request_number,
                        'material_issue',
                        $transaction->id,
                        $transaction->material_request_number
                    );

                    $processedItems[] = [
                        'item_id' => $itemId,
                        'quantity' => $qtyFromThisItem,
                        'source_reference' => $focInventoryItem->source_reference ?? 'FOC-' . str_pad($focInventoryItem->id, 6, '0', STR_PAD_LEFT),
                        'fulfillment_id' => $fulfillment->id,
                        'foc_inventory_id' => $focInventoryItem->id
                    ];
                }

                // Update outgoing item quantities
                $outgoingItem = OutgoingItem::where('outgoing_transaction_id', $transaction->id)
                    ->where('item_id', $itemId)
                    ->first();

                if ($outgoingItem) {
                    $outgoingItem->quantity_released += $quantityToTransfer;
                    $outgoingItem->save(); // quantity_balance is auto-calculated in the model's boot method
                }

                // Add to project inventory at zero cost (FOC)
                $this->updateProjectInventoryWithZeroCost($transaction->project_id, $itemId, $quantityToTransfer);

                Log::info("FOC Transfer processed", [
                    'transaction_id' => $transaction->id,
                    'item_id' => $itemId,
                    'quantity' => $quantityToTransfer,
                    'processed_items' => count($processedItems)
                ]);
            }

            return [
                'success' => true,
                'type' => 'foc_transfer',
                'items_processed' => count($processedItems),
                'total_quantity' => array_sum(array_column($processedItems, 'quantity')),
                'message' => 'FOC transfer completed successfully',
                'details' => $processedItems
            ];

        } catch (\Exception $e) {
            Log::error("FOC Transfer failed", [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'foc_items' => $focItems
            ]);

            throw $e;
        }
    }

    /**
     * Debug method to inspect FOC inventory data
     */
    public function debugFocInventory(Request $request)
    {
        $itemId = $request->input('item_id');

        // Get all FOC inventory records for debugging
        $query = \App\Models\FocInventory::with(['item']);

        if ($itemId) {
            $query->where('item_id', $itemId);
        }

        $allRecords = $query->get();

        $availableRecords = \App\Models\FocInventory::available()
            ->when($itemId, fn($q) => $q->where('item_id', $itemId))
            ->get();

        $summary = [
            'total_records' => $allRecords->count(),
            'available_records' => $availableRecords->count(),
            'status_breakdown' => $allRecords->groupBy('status')->map->count(),
            'quality_status_breakdown' => $allRecords->groupBy('quality_status')->map->count(),
            'records_with_quantity' => $allRecords->where('quantity_available', '>', 0)->count(),
            'all_records' => $allRecords->toArray(),
            'available_records_details' => $availableRecords->toArray()
        ];

        return response()->json($summary);
    }

    /**
     * Update project inventory with zero cost for FOC items
     */
    private function updateProjectInventoryWithZeroCost($projectId, $itemId, $quantity)
    {
        $existingProjectInventory = ProjectInventory::where('project_id', $projectId)
            ->where('item_id', $itemId)
            ->first();

        if ($existingProjectInventory) {
            // Add to existing inventory at zero cost impact
            $existingProjectInventory->quantity_available += $quantity;
            $existingProjectInventory->save();
        } else {
            // Create new project inventory with zero cost
            ProjectInventory::create([
                'project_id' => $projectId,
                'item_id' => $itemId,
                'quantity_available' => $quantity,
                'allocated_quantity' => 0,
                'unit_price' => 0, // FOC items have zero cost
                'total_value' => 0
            ]);
        }
    }
}