<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\ItemCategory;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ItemCategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.item-categories.view')->only(['index', 'show']);
        $this->middleware('warehouse.permission:warehouse.item-categories.create')->only(['create', 'store']);
        $this->middleware('warehouse.permission:warehouse.item-categories.edit')->only(['edit', 'update']);
        $this->middleware('warehouse.permission:warehouse.item-categories.delete')->only(['destroy']);
    }

    /**
     * Display a listing of item categories.
     */
    public function index(Request $request)
    {
        if ($request->expectsJson()) {
            return $this->getDataForTable($request);
        }

        // Get filter parameters
        $status = $request->get('status');
        $typeOfGoods = $request->get('type_of_goods');
        $search = $request->get('search');

        // Build query
        $query = ItemCategory::query();

        // Apply filters
        if (!empty($status)) {
            $query->where('status', $status);
        }

        if (!empty($typeOfGoods)) {
            $query->where('type_of_goods', $typeOfGoods);
        }

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('category_name', 'LIKE', "%{$search}%")
                  ->orWhere('category_code', 'LIKE', "%{$search}%")
                  ->orWhere('type_of_goods', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }

        // Get paginated results
        $categories = $query->orderBy('created_at', 'desc')
                           ->paginate($request->get('per_page', 25))
                           ->withQueryString();

        return view('warehouse.item-categories.index', compact('categories'));
    }

    /**
     * Get data for DataTable.
     */
    private function getDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        $query = ItemCategory::query();

        // Apply search
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('category_name', 'LIKE', "%{$search}%")
                  ->orWhere('category_code', 'LIKE', "%{$search}%")
                  ->orWhere('type_of_goods', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }

        // Apply status filter
        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        // Get total count
        $totalRecords = ItemCategory::count();
        $filteredRecords = $query->count();

        // Apply pagination
        $categories = $query->skip($start)
                           ->take($length)
                           ->orderBy('created_at', 'desc')
                           ->get();

        // Format data for DataTable
        $data = $categories->map(function ($category) {
            return [
                'id' => $category->id,
                'category_code' => $category->category_code,
                'category_name' => $category->category_name,
                'type_of_goods' => $category->type_of_goods,
                'description' => $category->description,
                'status' => $category->status,
                'items_count' => $category->active_items_count,
                'created_at' => $category->created_at->format('M d, Y H:i'),
                'created_by' => 'Admin', // You can add created_by field later
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
    }

    /**
     * Show the form for creating a new item category.
     */
    public function create()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.create
        return view('warehouse.item-categories.create');
    }

    /**
     * Store a newly created item category.
     */
    public function store(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.create

        $validatedData = $request->validate([
            'category_code' => 'nullable|string|max:20|unique:item_categories,category_code',
            'category_name' => 'required|string|max:255|unique:item_categories,category_name',
            'type_of_goods' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'status' => 'required|in:active,inactive',
        ]);

        try {
            $category = ItemCategory::create($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $category,
                    'message' => 'Item category created successfully'
                ], 201);
            }

            return redirect()->route('warehouse.item-categories.index')
                           ->with('success', 'Item category created successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified item category.
     */
    public function show($id)
    {
        $category = ItemCategory::with('items')->findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.view

        return view('warehouse.item-categories.show', compact('category'));
    }

    /**
     * Show the form for editing the specified item category.
     */
    public function edit($id)
    {
        $category = ItemCategory::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.edit

        return view('warehouse.item-categories.edit', compact('category'));
    }

    /**
     * Update the specified item category.
     */
    public function update(Request $request, $id)
    {
        $category = ItemCategory::findOrFail($id);

        if (!$category) {
            abort(404, 'Item category not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.edit

        $validatedData = $request->validate([
            'category_code' => [
                'nullable',
                'string',
                'max:20',
                Rule::unique('item_categories', 'category_code')->ignore($id)
            ],
            'category_name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('item_categories', 'category_name')->ignore($id)
            ],
            'type_of_goods' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'status' => 'required|in:active,inactive',
        ]);

        try {
            $category->update($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $category,
                    'message' => 'Item category updated successfully'
                ]);
            }

            return redirect()->route('warehouse.item-categories.show', $category)
                           ->with('success', 'Item category updated successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified item category.
     */
    public function destroy($id)
    {
        $category = ItemCategory::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.delete

        try {
            // Check if category has items
            if ($category->items()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category that has items associated with it.'
                ], 400);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Item category deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download the item category import template.
     */
    public function downloadTemplate()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.view

        try {
            $filePath = $this->generateImportTemplate();
            $filename = 'item_categories_import_template.xlsx';

            return response()->download($filePath, $filename)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Import item categories from Excel file.
     */
    public function import(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.create

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240' // 10MB max
        ]);

        try {
            $results = $this->importFromExcel($request->file('file'));

            return response()->json([
                'success' => true,
                'imported' => $results['success'],
                'updated' => 0, // Currently we only create new categories
                'skipped' => count($results['errors']),
                'data' => $results,
                'message' => "Import completed. {$results['success']} item categories imported successfully."
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 422);
        }
    }

    /**
     * Export item categories to Excel.
     */
    public function export(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.view

        $filters = $request->only(['status', 'type_of_goods']);

        try {
            $filePath = $this->exportToExcel($filters);
            $filename = basename($filePath);

            return response()->download($filePath, $filename)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate the import template Excel file.
     */
    private function generateImportTemplate()
    {
        $spreadsheet = new Spreadsheet();

        // Data Sheet
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Data');

        // Headers
        $headers = [
            'A1' => 'category_code',
            'B1' => 'category_description',
            'C1' => 'type_of_goods',
            'D1' => 'status'
        ];

        foreach ($headers as $cell => $header) {
            $sheet->setCellValue($cell, $header);
            $sheet->getStyle($cell)->getFont()->setBold(true);
        }

        // Sample data
        $sampleData = [
            2 => [
                'A' => 'ELE',
                'B' => 'Electronics',
                'C' => 'Consumer Electronics',
                'D' => 'active'
            ],
            3 => [
                'A' => 'CON',
                'B' => 'Construction Materials',
                'C' => 'Building Materials',
                'D' => 'active'
            ],
            4 => [
                'A' => 'OFF',
                'B' => 'Office Supplies',
                'C' => 'Administrative',
                'D' => 'active'
            ]
        ];

        foreach ($sampleData as $row => $data) {
            foreach ($data as $col => $value) {
                $sheet->setCellValue($col . $row, $value);
            }
        }

        // Add comments/notes
        $sheet->getComment('A1')->getText()->createTextRun('Optional. Unique category code (max 20 characters). Auto-generated if empty.');
        $sheet->getComment('B1')->getText()->createTextRun('Required. Unique category description/name (max 255 characters)');
        $sheet->getComment('C1')->getText()->createTextRun('Required. Type of goods - will be auto-created if not exists (max 255 characters)');
        $sheet->getComment('D1')->getText()->createTextRun('Required. Status: active or inactive');

        // Auto-size columns
        foreach (range('A', 'D') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Instructions Sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');

        $instructions = [
            'ITEM CATEGORIES IMPORT INSTRUCTIONS',
            '',
            '1. REQUIRED FIELDS:',
            '   - category_description: Unique category name/description (required)',
            '   - type_of_goods: Type of goods (required, auto-created if not exists)',
            '   - status: active or inactive (required)',
            '',
            '2. OPTIONAL FIELDS:',
            '   - category_code: Unique category code (auto-generated if empty)',
            '',
            '3. VALIDATION RULES:',
            '   - Category descriptions and codes must be unique',
            '   - Status must be either "active" or "inactive"',
            '   - Maximum 20 characters for category_code',
            '   - Maximum 255 characters for category_description and type_of_goods',
            '',
            '4. DATA FORMAT:',
            '   - Do not modify the header row',
            '   - Delete the sample data rows before importing your data',
            '   - Ensure proper spelling for status values',
            '',
            '5. IMPORT PROCESS:',
            '   - Upload only .xlsx, .xls, or .csv files',
            '   - Maximum file size: 10MB',
            '   - Duplicate category names will be skipped',
            '   - Invalid rows will be reported in import results',
            '',
            '6. EXAMPLE DATA:',
            '   Electronics | Consumer Electronics | Electronic devices | active',
            '   Construction | Building Materials | Construction materials | active',
            '   Office Supplies | Administrative | Office supplies | inactive'
        ];

        foreach ($instructions as $index => $instruction) {
            $instructionsSheet->setCellValue('A' . ($index + 1), $instruction);
            if ($index === 0) {
                $instructionsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
            }
        }

        $instructionsSheet->getColumnDimension('A')->setWidth(80);

        // Set active sheet back to data
        $spreadsheet->setActiveSheetIndex(0);

        // Save file
        $tempDir = storage_path('app/temp');
        if (!file_exists($tempDir)) {
            mkdir($tempDir, 0755, true);
        }

        $fileName = 'item_categories_import_template.xlsx';
        $filePath = $tempDir . '/' . $fileName;

        $writer = new Xlsx($spreadsheet);
        $writer->save($filePath);

        return $filePath;
    }

    /**
     * Import item categories from Excel file.
     */
    private function importFromExcel($file)
    {
        $spreadsheet = IOFactory::load($file->getPathname());
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        // Remove header row
        array_shift($rows);

        $imported = 0;
        $errors = [];

        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2; // +2 because we removed header and arrays are 0-indexed

            // Skip empty rows
            if (empty(array_filter($row))) {
                continue;
            }

            try {
                // Map columns
                $categoryData = [
                    'category_code' => trim($row[0] ?? '') ?: null, // Auto-generate if empty
                    'category_name' => trim($row[1] ?? ''), // This is category_description from import
                    'type_of_goods' => trim($row[2] ?? ''),
                    'status' => trim($row[3] ?? 'active')
                ];

                // Validate required fields
                if (empty($categoryData['category_name'])) {
                    $errors[] = "Row {$rowNumber}: category_name is required";
                    continue;
                }

                if (empty($categoryData['type_of_goods'])) {
                    $errors[] = "Row {$rowNumber}: type_of_goods is required";
                    continue;
                }

                // Validate status
                if (!in_array($categoryData['status'], ['active', 'inactive'])) {
                    $errors[] = "Row {$rowNumber}: status must be 'active' or 'inactive'";
                    continue;
                }

                // Check for duplicate category name
                if (ItemCategory::where('category_name', $categoryData['category_name'])->exists()) {
                    $errors[] = "Row {$rowNumber}: Category '{$categoryData['category_name']}' already exists";
                    continue;
                }

                // Check for duplicate category code if provided
                if (!empty($categoryData['category_code']) && ItemCategory::where('category_code', $categoryData['category_code'])->exists()) {
                    $errors[] = "Row {$rowNumber}: Category code '{$categoryData['category_code']}' already exists";
                    continue;
                }

                // Auto-create type_of_goods if it doesn't exist
                if (!empty($categoryData['type_of_goods'])) {
                    $existingType = \App\Models\Warehouse\ItemType::where('type_name', $categoryData['type_of_goods'])->first();
                    if (!$existingType) {
                        // Create new ItemType
                        \App\Models\Warehouse\ItemType::create([
                            'type_name' => $categoryData['type_of_goods'],
                            'type_code' => strtoupper(substr(preg_replace('/[^A-Za-z]/', '', $categoryData['type_of_goods']), 0, 3)),
                            'description' => 'Auto-created during category import',
                            'status' => 'active'
                        ]);
                    }
                }

                // Create category
                ItemCategory::create($categoryData);
                $imported++;

            } catch (\Exception $e) {
                $errors[] = "Row {$rowNumber}: " . $e->getMessage();
            }
        }

        return [
            'success' => $imported,
            'errors' => $errors
        ];
    }

    /**
     * Export item categories to Excel.
     */
    private function exportToExcel($filters = [])
    {
        $query = ItemCategory::query();

        // Apply filters
        if (isset($filters['status']) && !empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['type_of_goods']) && !empty($filters['type_of_goods'])) {
            $query->where('type_of_goods', 'LIKE', '%' . $filters['type_of_goods'] . '%');
        }

        $categories = $query->orderBy('category_name')->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Item Categories');

        // Headers
        $headers = ['Category Name', 'Type of Goods', 'Description', 'Status', 'Items Count', 'Created Date'];
        foreach ($headers as $index => $header) {
            $column = chr(65 + $index); // A, B, C, etc.
            $sheet->setCellValue($column . '1', $header);
            $sheet->getStyle($column . '1')->getFont()->setBold(true);
        }

        // Data
        foreach ($categories as $index => $category) {
            $row = $index + 2;
            $sheet->setCellValue('A' . $row, $category->category_name);
            $sheet->setCellValue('B' . $row, $category->type_of_goods);
            $sheet->setCellValue('C' . $row, $category->description);
            $sheet->setCellValue('D' . $row, ucfirst($category->status));
            $sheet->setCellValue('E' . $row, $category->active_items_count);
            $sheet->setCellValue('F' . $row, $category->created_at->format('Y-m-d H:i:s'));
        }

        // Auto-size columns
        foreach (range('A', 'F') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Save file
        $tempDir = storage_path('app/temp');
        if (!file_exists($tempDir)) {
            mkdir($tempDir, 0755, true);
        }

        $fileName = 'item_categories_export_' . date('Y-m-d_H-i-s') . '.xlsx';
        $filePath = $tempDir . '/' . $fileName;

        $writer = new Xlsx($spreadsheet);
        $writer->save($filePath);

        return $filePath;
    }
}