<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\DirectDelivery;
use App\Models\DirectDeliveryItem;
use App\Models\Warehouse\Supplier;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class DirectDeliveryController extends Controller
{
    /**
     * Display a listing of direct deliveries.
     */
    public function index(Request $request)
    {
        $query = DirectDelivery::with([
            'supplier', // This can be null for material requests
            'project',
            'projectDivision',
            'receivedBy',
            'items',
            'materialRequest' // Include material request relationship
        ])->latest('created_at'); // LIFO order - newest entries first

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('delivery_type')) {
            $query->where('delivery_type', $request->delivery_type);
        }

        if ($request->filled('date_from')) {
            $query->where('delivery_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('delivery_date', '<=', $request->date_to);
        }

        $directDeliveries = $query->paginate(15);

        // Get filter options
        $suppliers = Supplier::orderBy('name')->get();
        $projects = Project::orderBy('project_name')->get();

        return view('warehouse.direct-deliveries.index', compact(
            'directDeliveries', 'suppliers', 'projects'
        ));
    }

    /**
     * Show the form for creating a new direct delivery.
     */
    public function create()
    {
        $suppliers = Supplier::orderBy('name')->get();
        $projects = Project::orderBy('project_name')->get();
        $items = Item::orderBy('item_description')->get();

        return view('warehouse.direct-deliveries.create', compact(
            'suppliers', 'projects', 'items'
        ));
    }

    /**
     * Store a newly created direct delivery.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'nullable|exists:suppliers,id',
            'project_id' => 'required|exists:projects,id',
            'project_division_id' => 'nullable|exists:project_divisions,id',
            'delivery_date' => 'required|date',
            'order_date' => 'nullable|date',
            'delivery_type' => 'required|in:purchase_order,direct_purchase,emergency',
            'operation_type' => 'required|in:supplier_delivery,site_return,inter_division_transfer,emergency_procurement',
            'project_phase' => 'nullable|string|max:255',
            'site_location' => 'nullable|string|max:255',
            'lpo_number' => 'nullable|string|max:255',
            'lpo_status' => 'nullable|in:pending,approved,partially_delivered,completed,cancelled',
            'lpo_budget_amount' => 'nullable|numeric|min:0',
            'purchase_order_number' => 'nullable|string|max:255',
            'supplier_delivery_note' => 'nullable|string|max:255',
            'delivered_by' => 'nullable|string|max:255',
            'vehicle_number' => 'nullable|string|max:255',
            'driver_contact' => 'nullable|string|max:255',
            'delivery_notes' => 'nullable|string',
            'delivery_instructions' => 'nullable|string',
            'requires_approval' => 'boolean',
            'requires_inspection' => 'boolean',
            'inspector_id' => 'nullable|exists:users,id',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.ordered_quantity' => 'required|numeric|min:0.01',
            'items.*.delivered_quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.unit_of_measure' => 'nullable|string|max:50',
            'items.*.batch_number' => 'nullable|string|max:255',
            'items.*.expiry_date' => 'nullable|date',
            'items.*.manufacturer' => 'nullable|string|max:255',
            'items.*.specifications' => 'nullable|string'
        ]);

        DB::beginTransaction();

        try {
            // Create direct delivery
            $directDelivery = DirectDelivery::create($validated);

            // Create delivery items
            foreach ($validated['items'] as $itemData) {
                $totalPrice = $itemData['delivered_quantity'] * $itemData['unit_price'];

                DirectDeliveryItem::create([
                    'direct_delivery_id' => $directDelivery->id,
                    'item_id' => $itemData['item_id'],
                    'ordered_quantity' => $itemData['ordered_quantity'],
                    'delivered_quantity' => $itemData['delivered_quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'total_price' => $totalPrice,
                    'unit_of_measure' => $itemData['unit_of_measure'] ?? null,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                    'manufacturer' => $itemData['manufacturer'] ?? null,
                    'specifications' => $itemData['specifications'] ?? null
                ]);
            }

            // Calculate total
            $directDelivery->calculateTotal();

            DB::commit();

            return redirect()->route('warehouse.direct-deliveries.show', $directDelivery)
                ->with('success', 'Direct delivery created successfully.');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withInput()
                ->with('error', 'Error creating direct delivery: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified direct delivery.
     */
    public function show(DirectDelivery $directDelivery)
    {
        $directDelivery->load([
            'supplier',
            'project',
            'projectDivision',
            'receivedBy',
            'approvedBy',
            'items.item',
            'items.qualityCheckedBy'
        ]);

        return view('warehouse.direct-deliveries.show', compact('directDelivery'));
    }

    /**
     * Show the form for editing the specified direct delivery.
     */
    public function edit(DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canEdit()) {
            return back()->with('error', 'This delivery cannot be edited.');
        }

        $suppliers = Supplier::orderBy('name')->get();
        $projects = Project::orderBy('project_name')->get();
        $items = Item::orderBy('item_description')->get();
        $users = \App\Models\User::whereHas('role', function($query) {
            $query->whereIn('name', ['warehouse_manager', 'quality_inspector', 'admin']);
        })->orderBy('name')->get();

        $directDelivery->load(['items.item', 'supplier', 'project', 'materialRequest']);

        return view('warehouse.direct-deliveries.edit', compact(
            'directDelivery', 'suppliers', 'projects', 'items', 'users'
        ));
    }

    /**
     * Update the specified direct delivery.
     */
    public function update(Request $request, DirectDelivery $directDelivery)
    {
        // Debug logging
        \Log::info('DirectDelivery Update Request', [
            'delivery_id' => $directDelivery->id,
            'action' => $request->input('action'),
            'request_data' => $request->all()
        ]);

        if (!$directDelivery->canEdit()) {
            return back()->with('error', 'This delivery cannot be edited.');
        }

        // Check if supplier is already set and prevent changes
        if ($directDelivery->supplier_id && $request->input('supplier_id') != $directDelivery->supplier_id) {
            return back()->withInput()->withErrors([
                'supplier_id' => 'Supplier cannot be changed after initial selection.'
            ]);
        }

        // Check if LPO number is already set and prevent changes
        if ($directDelivery->lpo_number && $request->input('lpo_number') !== $directDelivery->lpo_number) {
            return back()->withInput()->withErrors([
                'lpo_number' => 'LPO Number cannot be changed after initial entry.'
            ]);
        }

        \Log::info('Starting validation', ['delivery_id' => $directDelivery->id]);

        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'project_id' => 'required|exists:projects,id',
            'project_division_id' => 'nullable|exists:project_divisions,id',
            'delivery_date' => 'required|date',
            'order_date' => 'nullable|date',
            'delivery_type' => 'nullable|in:purchase_order,direct_purchase,emergency',
            'operation_type' => 'nullable|in:supplier_delivery,site_return,inter_division_transfer,emergency_procurement',
            'project_phase' => 'nullable|string|max:255',
            'site_location' => 'nullable|string|max:255',
            'lpo_number' => 'nullable|string|max:255',
            'lpo_status' => 'nullable|in:pending,approved,partially_delivered,completed,cancelled',
            'lpo_budget_amount' => 'nullable|numeric|min:0',
            'purchase_order_number' => 'nullable|string|max:255',
            'supplier_delivery_note' => 'nullable|string|max:255',
            'delivered_by' => 'nullable|string|max:255',
            'vehicle_number' => 'nullable|string|max:255',
            'driver_contact' => 'nullable|string|max:255',
            'delivery_notes' => 'nullable|string',
            'delivery_instructions' => 'nullable|string',
            'requires_approval' => 'boolean',
            'requires_inspection' => 'boolean',
            'inspector_id' => 'nullable|exists:users,id',
            'inspection_result' => 'nullable|in:passed,failed,conditional,pending',
            'inspection_notes' => 'nullable|string',
            'regulatory_compliant' => 'boolean',
            'items' => 'array',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.ordered_quantity' => 'required|numeric|min:0',
            'items.*.delivered_quantity' => 'required|numeric|min:0',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.total_price' => 'nullable|numeric|min:0'
        ]);

        \Log::info('Validation passed', ['delivery_id' => $directDelivery->id]);

        // Set default values for missing fields
        if (empty($validated['delivery_type'])) {
            $validated['delivery_type'] = 'purchase_order';
        }
        if (empty($validated['operation_type'])) {
            $validated['operation_type'] = 'supplier_delivery';
        }

        DB::beginTransaction();
        try {
            \Log::info('Starting database transaction', ['delivery_id' => $directDelivery->id]);

            // Update the direct delivery
            $deliveryData = $validated;
            unset($deliveryData['items']);
            \Log::info('Updating delivery data', ['delivery_id' => $directDelivery->id, 'data' => $deliveryData]);

            // Try to update and log the result
            $updateResult = $directDelivery->update($deliveryData);
            \Log::info('Delivery update result', ['delivery_id' => $directDelivery->id, 'result' => $updateResult]);

            // Refresh the model to get the latest data
            $directDelivery->refresh();
            \Log::info('Delivery data after update', ['delivery_id' => $directDelivery->id, 'updated_at' => $directDelivery->updated_at]);

            // Handle items if provided
            if (isset($validated['items'])) {
                // Validate business rules for existing items
                $existingItems = $directDelivery->items()->with('item')->get()->keyBy('id');

                foreach ($validated['items'] as $index => $itemData) {
                    // Find matching existing item by item_id
                    $existingItem = $existingItems->firstWhere('item_id', $itemData['item_id']);

                    if ($existingItem) {
                        $newDeliveredQty = (float) $itemData['delivered_quantity'];
                        $existingDeliveredQty = (float) $existingItem->delivered_quantity;
                        $newOrderedQty = (float) $itemData['ordered_quantity'];
                        $existingOrderedQty = (float) $existingItem->ordered_quantity;

                        \Log::info('Validating item business rules', [
                            'item_id' => $itemData['item_id'],
                            'new_delivered' => $newDeliveredQty,
                            'existing_delivered' => $existingDeliveredQty,
                            'new_ordered' => $newOrderedQty,
                            'existing_ordered' => $existingOrderedQty
                        ]);

                        // Rule 1: Cannot decrease delivered quantity
                        if ($existingDeliveredQty > 0 && $newDeliveredQty < $existingDeliveredQty) {
                            \Log::warning('Validation failed: Cannot decrease delivered quantity', [
                                'item_id' => $itemData['item_id'],
                                'new' => $newDeliveredQty,
                                'existing' => $existingDeliveredQty
                            ]);
                            return back()->withInput()->withErrors([
                                "items.{$index}.delivered_quantity" => "Cannot decrease delivered quantity below {$existingDeliveredQty} for {$existingItem->item->item_description}"
                            ]);
                        }

                        // Rule 2: Cannot exceed ordered quantity
                        if ($newDeliveredQty > $newOrderedQty) {
                            \Log::warning('Validation failed: Cannot exceed ordered quantity', [
                                'item_id' => $itemData['item_id'],
                                'new_delivered' => $newDeliveredQty,
                                'new_ordered' => $newOrderedQty
                            ]);
                            return back()->withInput()->withErrors([
                                "items.{$index}.delivered_quantity" => "Delivered quantity cannot exceed ordered quantity of {$newOrderedQty} for {$existingItem->item->item_description}"
                            ]);
                        }

                        // Rule 3: Cannot reduce ordered quantity unless it matches delivered quantity (force close)
                        if ($newOrderedQty < $existingOrderedQty && $newOrderedQty != $newDeliveredQty) {
                            \Log::warning('Validation failed: Cannot reduce ordered quantity unless force closing', [
                                'item_id' => $itemData['item_id'],
                                'new_ordered' => $newOrderedQty,
                                'existing_ordered' => $existingOrderedQty,
                                'new_delivered' => $newDeliveredQty
                            ]);
                            return back()->withInput()->withErrors([
                                "items.{$index}.ordered_quantity" => "Cannot reduce ordered quantity from {$existingOrderedQty} to {$newOrderedQty} unless force closing (ordered = delivered) for {$existingItem->item->item_description}"
                            ]);
                        }

                        // Rule 4: Cannot change unit price for partially delivered items
                        if ($existingItem->delivered_quantity > 0 && $itemData['unit_price'] != $existingItem->unit_price) {
                            return back()->withInput()->withErrors([
                                "items.{$index}.unit_price" => "Cannot change unit price for partially delivered item: {$existingItem->item->item_description}"
                            ]);
                        }
                    }
                }

                // Delete existing items and recreate
                \Log::info('Deleting existing items', ['delivery_id' => $directDelivery->id]);
                $directDelivery->items()->delete();
                \Log::info('Existing items deleted', ['delivery_id' => $directDelivery->id]);

                \Log::info('Creating new items', ['delivery_id' => $directDelivery->id, 'item_count' => count($validated['items'])]);
                foreach ($validated['items'] as $index => $itemData) {
                    $totalPrice = isset($itemData['total_price']) ? $itemData['total_price'] : ($itemData['delivered_quantity'] * $itemData['unit_price']);

                    $newItem = DirectDeliveryItem::create([
                        'direct_delivery_id' => $directDelivery->id,
                        'item_id' => $itemData['item_id'],
                        'ordered_quantity' => $itemData['ordered_quantity'],
                        'delivered_quantity' => $itemData['delivered_quantity'],
                        'received_quantity' => 0,
                        'rejected_quantity' => 0,
                        'unit_price' => $itemData['unit_price'],
                        'total_price' => $totalPrice,
                        'quality_status' => 'pending',
                    ]);

                    \Log::info('Created new item', [
                        'delivery_id' => $directDelivery->id,
                        'item_index' => $index,
                        'item_id' => $itemData['item_id'],
                        'created_item_id' => $newItem->id,
                        'delivered_quantity' => $itemData['delivered_quantity']
                    ]);
                }
                \Log::info('All items created successfully', ['delivery_id' => $directDelivery->id]);

                // Recalculate total
                $directDelivery->calculateTotal();

                // Update delivery status based on item quantities
                $directDelivery->updateDeliveryStatus();
            }

            // Handle workflow status update before committing
            $message = 'Direct delivery updated successfully.';
            $action = $request->input('action');
            \Log::info('Processing action', ['action' => $action, 'delivery_id' => $directDelivery->id]);

            if ($action === 'submit') {
                \Log::info('Updating workflow status to pending_inspection', ['delivery_id' => $directDelivery->id]);
                $directDelivery->update(['workflow_status' => 'pending_inspection']);
                $message = 'Direct delivery submitted for processing successfully.';
            }

            DB::commit();

            \Log::info('Transaction committed successfully', ['delivery_id' => $directDelivery->id, 'message' => $message]);

            return redirect()->route('warehouse.direct-deliveries.show', $directDelivery)
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('DirectDelivery Update Failed', [
                'delivery_id' => $directDelivery->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            return back()->withErrors(['error' => 'Failed to update direct delivery: ' . $e->getMessage()]);
        }
    }

    /**
     * Remove the specified direct delivery.
     */
    public function destroy(DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canEdit()) {
            return back()->with('error', 'This delivery cannot be deleted.');
        }

        $directDelivery->delete();

        return redirect()->route('warehouse.direct-deliveries.index')
            ->with('success', 'Direct delivery deleted successfully.');
    }

    /**
     * Show the receipt form for the delivery.
     */
    public function receipt(DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canReceive()) {
            return back()->with('error', 'This delivery cannot be received.');
        }

        $directDelivery->load(['supplier', 'project', 'items.item']);

        return view('warehouse.direct-deliveries.receipt', compact('directDelivery'));
    }

    /**
     * Process the receipt of the delivery.
     */
    public function processReceipt(Request $request, DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canReceive()) {
            return back()->with('error', 'This delivery cannot be received.');
        }

        $validated = $request->validate([
            'receipt_notes' => 'nullable|string',
            'items' => 'required|array',
            'items.*.item_id' => 'required|exists:direct_delivery_items,id',
            'items.*.received_quantity' => 'required|numeric|min:0',
            'items.*.rejected_quantity' => 'nullable|numeric|min:0',
            'items.*.quality_status' => 'required|in:pending,passed,failed,partial',
            'items.*.quality_notes' => 'nullable|string',
            'items.*.storage_location' => 'nullable|string|max:255',
            'items.*.receipt_notes' => 'nullable|string'
        ]);

        DB::beginTransaction();

        try {
            $fullyReceived = true;

            foreach ($validated['items'] as $itemData) {
                $deliveryItem = DirectDeliveryItem::find($itemData['item_id']);

                $deliveryItem->update([
                    'received_quantity' => $itemData['received_quantity'],
                    'rejected_quantity' => $itemData['rejected_quantity'] ?? 0,
                    'quality_status' => $itemData['quality_status'],
                    'quality_notes' => $itemData['quality_notes'],
                    'storage_location' => $itemData['storage_location'],
                    'receipt_notes' => $itemData['receipt_notes'],
                    'quality_checked_at' => now(),
                    'quality_checked_by' => Auth::id()
                ]);

                // Calculate total price based on received quantity
                $deliveryItem->calculateTotal();

                // Check if fully received
                if ($deliveryItem->received_quantity < $deliveryItem->delivered_quantity) {
                    $fullyReceived = false;
                }
            }

            // Update delivery status
            if ($fullyReceived) {
                $directDelivery->markAsReceived(Auth::id(), $validated['receipt_notes']);
            } else {
                $directDelivery->markAsPartiallyReceived(Auth::id(), $validated['receipt_notes']);
            }

            DB::commit();

            return redirect()->route('warehouse.direct-deliveries.show', $directDelivery)
                ->with('success', 'Delivery receipt processed successfully.');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withInput()
                ->with('error', 'Error processing receipt: ' . $e->getMessage());
        }
    }

    /**
     * Approve the delivery.
     */
    public function approve(Request $request, DirectDelivery $directDelivery)
    {
        if (!$directDelivery->needsApproval()) {
            return back()->with('error', 'This delivery does not require approval or is already approved.');
        }

        $validated = $request->validate([
            'approval_notes' => 'nullable|string'
        ]);

        $directDelivery->approve(Auth::id(), $validated['approval_notes']);

        return back()->with('success', 'Delivery approved successfully.');
    }

    /**
     * Get project divisions for a specific project (AJAX).
     */
    public function getProjectDivisions($projectId)
    {
        $divisions = ProjectDivision::where('project_id', $projectId)
            ->orderBy('division_name')
            ->get(['id', 'division_name']);

        return response()->json($divisions);
    }

    /**
     * Show site return form for the delivery.
     */
    public function siteReturn(DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canBeReturned()) {
            return back()->with('error', 'This delivery cannot be returned.');
        }

        $directDelivery->load(['supplier', 'project', 'projectDivision', 'items.item']);
        $returnableItems = $directDelivery->getReturnableItems();

        return view('warehouse.direct-deliveries.site-return', compact('directDelivery', 'returnableItems'));
    }

    /**
     * Process the site return.
     */
    public function processSiteReturn(Request $request, DirectDelivery $directDelivery)
    {
        if (!$directDelivery->canBeReturned()) {
            return back()->with('error', 'This delivery cannot be returned.');
        }

        $validated = $request->validate([
            'return_date' => 'required|date',
            'return_reason' => 'required|string|max:255',
            'force_close' => 'boolean',
            'force_close_reason' => 'nullable|string|max:500',
            'return_notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.selected' => 'nullable|boolean',
            'items.*.item_id' => 'required|exists:direct_delivery_items,id',
            'items.*.return_quantity' => 'nullable|numeric|min:0.01',
            'items.*.return_reason' => 'nullable|string|max:255',
            'items.*.condition' => 'nullable|in:good,damaged,defective,expired',
            'items.*.return_notes' => 'nullable|string'
        ]);

        // Filter only selected items and validate their required fields
        $selectedItems = [];
        foreach ($validated['items'] as $index => $itemData) {
            if (!empty($itemData['selected'])) {
                // Validate required fields for selected items
                if (empty($itemData['return_quantity']) || $itemData['return_quantity'] <= 0) {
                    return back()->withInput()->withErrors(["items.{$index}.return_quantity" => "Return quantity is required for selected items."]);
                }
                if (empty($itemData['condition'])) {
                    return back()->withInput()->withErrors(["items.{$index}.condition" => "Condition is required for selected items."]);
                }
                $selectedItems[] = $itemData;
            }
        }

        if (empty($selectedItems)) {
            return back()->withInput()->withErrors(['items' => 'Please select at least one item to return.']);
        }

        $validated['items'] = $selectedItems;

        DB::beginTransaction();

        try {
            // Create site return operation
            $incomingOperation = $directDelivery->createSiteReturn([
                'return_date' => $validated['return_date'],
                'force_close' => $validated['force_close'] ?? false,
                'force_close_reason' => $validated['force_close_reason'] ?? null,
                'division' => $directDelivery->projectDivision->division_name ?? null,
            ]);

            $totalReturnValue = 0;

            // Process each returned item
            foreach ($validated['items'] as $itemData) {
                $deliveryItem = DirectDeliveryItem::find($itemData['item_id']);
                $returnQuantity = $itemData['return_quantity'];

                // Validate return quantity
                if ($returnQuantity > $deliveryItem->getAvailableForReturnQuantity()) {
                    throw new \Exception("Return quantity for item {$deliveryItem->item->item_description} exceeds available quantity.");
                }

                // Create incoming operation item
                $incomingOperationItem = IncomingOperationItem::create([
                    'incoming_operation_id' => $incomingOperation->id,
                    'item_id' => $deliveryItem->item_id,

                    // Quantity information
                    'quantity_ordered' => 0,
                    'quantity_delivered' => $returnQuantity,
                    'quantity_accepted' => $returnQuantity, // Auto-accept for now
                    'quantity_rejected' => 0,
                    'unit_of_measure' => $deliveryItem->unit_of_measure,

                    // Pricing (same as original delivery)
                    'unit_price' => $deliveryItem->unit_price,
                    'total_price' => $returnQuantity * $deliveryItem->unit_price,
                    'currency' => $directDelivery->currency,
                    'unit_price_aed' => $deliveryItem->unit_price, // Assuming AED for now
                    'total_price_aed' => $returnQuantity * $deliveryItem->unit_price,

                    // Batch and quality information
                    'batch_number' => $deliveryItem->batch_number,
                    'expiry_date' => $deliveryItem->expiry_date,
                    'manufacturer' => $deliveryItem->manufacturer,

                    // Quality control
                    'quality_status' => 'pending',
                    'quality_notes' => $itemData['return_notes'],
                    'rejection_reason' => $itemData['return_reason'],

                    // Storage
                    'storage_location' => $deliveryItem->storage_location,
                ]);

                // Update delivery item return tracking
                $deliveryItem->recordReturn($returnQuantity, $itemData['return_reason']);

                $totalReturnValue += $returnQuantity * $deliveryItem->unit_price;
            }

            // Update operation total
            $incomingOperation->update(['total_amount' => $totalReturnValue]);

            DB::commit();

            return redirect()->route('warehouse.direct-deliveries.show', $directDelivery)
                ->with('success', "Site return {$incomingOperation->operation_number} created successfully.");

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withInput()
                ->with('error', 'Error processing site return: ' . $e->getMessage());
        }
    }

    /**
     * Print direct delivery document.
     */
    public function print(DirectDelivery $directDelivery)
    {
        $directDelivery->load([
            'supplier',
            'project',
            'projectDivision',
            'items.item'
        ]);

        // Create a delivery note object that matches our template structure
        $deliveryNote = (object) [
            'outgoingTransaction' => (object) [
                'mr_number' => $directDelivery->delivery_number,
            ],
            'delivered_to' => $directDelivery->project ?
                $directDelivery->project->project_name : 'N/A',
            'generated_date' => $directDelivery->delivery_date,
            'project' => $directDelivery->project ? (object) [
                'project_number' => $directDelivery->project->project_number ?:
                    $directDelivery->project->project_name
            ] : null,
            'items' => $directDelivery->items->map(function($item) {
                return (object) [
                    'item' => (object) [
                        'item_name' => $item->item->item_description,
                        'unit' => $item->unit_of_measure ?: 'pcs'
                    ],
                    'quantity_released' => $item->quantity_delivered ?: $item->quantity_expected,
                    'remarks' => $item->notes ?: ''
                ];
            })
        ];

        return view('warehouse.delivery-notes.template', compact('deliveryNote'));
    }
}